#ifndef FEATURE_EXTRACTOR_H
#define FEATURE_EXTRACTOR_H

#include <vector>
#include <complex>
#include <iostream>
#include "SignalFeatures.h"

enum class FilterType {HighPass, BandPass};

struct FilterCoefficients
{
        FilterType type;
        int lowcut;
        int highcut;
        std::pair<std::vector<float>, std::vector<float> > coeffs;
};

// 信号特征提取器类，实现完整的信号处理流程
class FeatureExtractor
{
private:
    int m_fs;      // 采样频率
    int m_cutoff;  //高通滤波参数
    int m_lowcut;  //带通滤波参数
    int m_highcut; //带通滤波参数
    int m_order;

public:
    // 构造函数，初始化采样频率
    FeatureExtractor();
    FeatureExtractor(int sample_freq, int cutoff, int lowcut, int highcut, int order);

    // 禁止拷贝构造和赋值操作
    FeatureExtractor(const FeatureExtractor&) = delete;
    FeatureExtractor& operator=(const FeatureExtractor&) = delete;

    // 析构函数，清理资源
    ~FeatureExtractor();

    // 提取信号特征的主入口函数
    SignalFeatures extract_features(const std::vector<float>& signal);

private:
    // Butterworth滤波器设计（直接使用预定义系数）
    std::pair<std::vector<float>, std::vector<float>> get_coeffs(FilterType type, int lowcut, int highcut, int order);

    // 零相位滤波（与scipy.signal.filtfilt一致）
    std::vector<float> filtfilt(const std::vector<float>& b,
                               const std::vector<float>& a,
                               const std::vector<float>& x);

    // 高通滤波（使用Butterworth + filtfilt）
    std::vector<float> butterworth_highpass_filter(const std::vector<float>& signal, int cutoff, int order);

    // 带通滤波（使用Butterworth + filtfilt）
    std::vector<float> butterworth_bandpass_filter(const std::vector<float>& signal, int lowcut, int highcut, int order);

    // 希尔伯特变换提取包络
    std::vector<float> hilbert_transform(const std::vector<float>& signal);

    // 计算包络谱（与Python实现一致）
    std::pair<std::vector<float>, std::vector<float>> compute_envelope_spectrum(const std::vector<float>& envelope);

    // 计算FFT
    std::vector<std::complex<float>> compute_fft(const std::vector<float>& signal);
    std::vector<std::complex<float>> compute_ifft(const std::vector<std::complex<float>>& freq_domain);

    // 提取时域特征
    bool extract_time_domain_features(const std::vector<float>& signal, SignalFeatures& features);
};

#endif
