#include "VibrationDataReader.h"
#include <QDebug>
#include <QDataStream>

VibrationDataReader::VibrationDataReader(QObject *parent)
    : QObject(parent)
    , m_dataBlocks(0)
    , m_totalBytes(0)
{
}

bool VibrationDataReader::readFile(const QString &filename)
{
    QFile file(filename);
    if (!file.open(QIODevice::ReadOnly)) {
        qWarning() << "无法打开文件:" << filename;
        return false;
    }

    // 获取文件大小
    m_totalBytes = file.size();
    qDebug() << "文件大小:" << m_totalBytes << "字节";

    // 计算数据块数量 (每个数据块240字节 = 20组数据)
    m_dataBlocks = m_totalBytes / 240;
    int remainder = m_totalBytes % 240;

    if (remainder != 0) {
        qWarning() << "警告: 文件大小不是240字节的整数倍，剩余" << remainder << "字节";
        // 不直接返回false，尝试读取完整的数据块
    }

    // 计算总数据组数
    int totalGroups = m_dataBlocks * 20;
    qDebug() << "数据块数量:" << m_dataBlocks << "，总数据组数:" << totalGroups;

    // 清空之前的数据
    m_vibrationData.clear();
    m_vibrationData.reserve(totalGroups);

    // 读取数据
    QDataStream stream(&file);
    stream.setByteOrder(QDataStream::LittleEndian);
    stream.setFloatingPointPrecision(QDataStream::SinglePrecision);

    // 读取所有数据块
    for (int block = 0; block < m_dataBlocks; ++block) {
        // 读取当前数据块的20组数据
        for (int i = 0; i < 20; ++i) {
            VibrationData data;
            stream >> data.x >> data.y >> data.z;

            if (stream.status() != QDataStream::Ok) {
                qWarning() << "读取第" << block + 1 << "块第" << i + 1 << "组数据时发生错误";
                file.close();
                return false;
            }

            m_vibrationData.append(data);
        }
    }

    file.close();
    qDebug() << "成功读取" << m_vibrationData.size() << "组振动数据";
    return true;
}


std::array<std::vector<float>, 3> VibrationDataReader::convertToRawFormat() const
{
    std::array<std::vector<float>, 3> raw;

    size_t dataCount = m_vibrationData.size();

    // 预分配内存
    raw[0].reserve(dataCount); // X轴数据
    raw[1].reserve(dataCount); // Y轴数据
    raw[2].reserve(dataCount); // Z轴数据

    // 转换数据
    for (const auto& vibData : m_vibrationData) {
        raw[0].push_back(vibData.x);
        raw[1].push_back(vibData.y);
        raw[2].push_back(vibData.z);
    }

    return raw;
}

bool VibrationDataReader::readFileRaw(const QString &filename)
{
    QFile file(filename);
    if (!file.open(QIODevice::ReadOnly)) {
        qWarning() << "无法打开文件:" << filename;
        return false;
    }

    QByteArray fileData = file.readAll();
    file.close();

    m_totalBytes = fileData.size();
    qDebug() << "文件大小:" << m_totalBytes << "字节";

    // 计算数据块数量
    m_dataBlocks = m_totalBytes / 240;
    int remainder = m_totalBytes % 240;

    if (remainder != 0) {
        qWarning() << "警告: 文件大小不是240字节的整数倍，剩余" << remainder << "字节";
    }

    int totalGroups = m_dataBlocks * 20;
    qDebug() << "数据块数量:" << m_dataBlocks << "，总数据组数:" << totalGroups;

    m_vibrationData.clear();
    m_vibrationData.reserve(totalGroups);

    // 手动解析字节数据
    const char *data = fileData.constData();

    for (int block = 0; block < m_dataBlocks; ++block) {
        int blockOffset = block * 240; // 每个数据块240字节

        for (int i = 0; i < 20; ++i) {
            VibrationData vibData;
            int groupOffset = blockOffset + i * 12; // 每组数据12字节

            if (groupOffset + 12 > m_totalBytes) {
                qWarning() << "数据越界，停止读取";
                break;
            }

            // 读取4字节浮点数
            memcpy(&vibData.x, data + groupOffset, 4);
            memcpy(&vibData.y, data + groupOffset + 4, 4);
            memcpy(&vibData.z, data + groupOffset + 8, 4);

            m_vibrationData.append(vibData);
        }
    }

    qDebug() << "成功读取" << m_vibrationData.size() << "组振动数据(原始方式)";
    return true;
}



void VibrationDataReader::printDataInfo() const
{
    qDebug() << "=== 振动数据信息 ===";
    qDebug() << "文件大小:" << m_totalBytes << "字节";
    qDebug() << "数据块数量:" << m_dataBlocks << "(每块240字节)";
    qDebug() << "总数据组数:" << m_vibrationData.size();

    // 只显示前几组数据，避免输出太多
    int displayCount = qMin(40, m_vibrationData.size());
    qDebug() << "前" << displayCount << "组数据:";

    for (int i = 0; i < displayCount; ++i) {
        const VibrationData &data = m_vibrationData.at(i);
        qDebug() << QString("  第%1组: X=%2, Y=%3, Z=%4")
                    .arg(i + 1, 3)
                    .arg(data.x, 10, 'f', 6)
                    .arg(data.y, 10, 'f', 6)
                    .arg(data.z, 10, 'f', 6);
    }

    if (m_vibrationData.size() > displayCount) {
        qDebug() << "  ... (还有" << m_vibrationData.size() - displayCount << "组数据未显示)";
    }
}
