﻿using CSRedis;
using Microsoft.Extensions.DependencyInjection;
using Siger.CommonUtil.Logging;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.Serialization.Formatters.Binary;
using System.Threading.Tasks;
namespace Siger.WeComApi.Common.cache
{
    /// <summary>
    /// 缓存工具
    /// </summary>
    public class CacheUtility
    {
        private static CacheUtility cacheUtility;
        private const string LicenseHash = "Licenses";
        /// <summary>
        /// 单例实现
        /// </summary>
        public static CacheUtility Instance
        {

            get
            {
                if (cacheUtility == null)
                {
                    cacheUtility = new CacheUtility();
                }
                return cacheUtility;
            }
        }

        private static ISigerLogger<CacheUtility> logger;

        private CacheUtility()
        {
            logger = ServiceContainer.Current.GetService<ISigerLogger<CacheUtility>>();
        }
        private CSRedisClient client;
        private CSRedisClient Client
        {
            get
            {
                try
                {
                    if (client == null)
                    {
                        var useOneDb = ConfigManager.GetValue("DbSetting", "UseOneDb", true);
                        if (useOneDb)
                        {
                            client = GetSingleRedis();
                        }
                        else
                        {
                            client = GetClusterRedis();
                        }
                        RedisHelper.Initialization(client);
                    }
                }
                catch (Exception e)
                {
                    throw new Exception("connect redis failed:" + e.Message);
                }

                return client;
            }
        }

        /// <summary>
        /// 单服务redis
        /// </summary>
        /// <returns></returns>
        private CSRedisClient GetSingleRedis()
        {
            var connectionStr = ConfigManager.GetValue("RedisSetting", "ConnectionString", string.Empty);
            var singleClient = new CSRedisClient(connectionStr);
            return singleClient;
        }

        /// <summary>
        /// redis集群
        /// </summary>
        /// <returns></returns>
        private CSRedisClient GetClusterRedis()
        {
            CSRedisClient redisClient = null;
            var redisHosts = ConfigManager.GetValue("DbClusterSetting", "RedisHost", "");
            if (!string.IsNullOrWhiteSpace(redisHosts))
            {
                var hosts = redisHosts.Split(';').ToArray();
                var connections = new string[hosts.Length];
                for (var i = 0; i < hosts.Length; i++)
                {
                    connections[i] = hosts[i];
                }
                redisClient = new CSRedisClient(null, connections);
            }
            return redisClient;
        }

        public async Task<T> GetAsync<T>(string key)
        {
            if (Client == null)
            {
                return default(T);
            }
            try
            {
                var temparr = await Client.GetAsync<byte[]>(key);
                return (T)DeserializeObject(temparr);
            }
            catch (Exception ex)
            {
                logger.Error(ex, "GetAsync error", key);
            }
            return default(T);
        }

        public async Task<string> GetHashAsync(string key, string field)
        {
            if (Client == null)
            {
                return string.Empty;
            }
            try
            {
                return await Client.HGetAsync(key, field);
            }
            catch (Exception ex)
            {
                logger.Error(ex, "", key, field);
            }
            return string.Empty;
        }

        public string GetLicense(string key)
        {
            return Client.HGet(LicenseHash, key);
        }

        public void AddLicenseEntity(string licName,string key,string lic)
        {
            Client.HSet(licName, key, lic);
        }

        public async Task<Dictionary<string,string>> HGetAllAsync(string key)
        {
            if (Client == null)
            {
                return new Dictionary<string, string>();
            }
            try
            {
                return await Client.HGetAllAsync(key);
            }
            catch (Exception ex)
            {
                logger.Error(ex, "", key);
            }
            return new Dictionary<string, string>();
        }

        public async Task<bool> SetHashAsync(string key, string field, object value)
        {
            if (Client == null)
            {
                return false;
            }
            try
            {
                return await Client.HSetAsync(key, field, value);
            }
            catch (Exception ex)
            {
                logger.Error(ex, new { key, field, value }, key, field);
            }
            return false;
        }

        public string GetString(string key)
        {
            if (Client == null)
            {
                return string.Empty;
            }
            try
            {
                return Client.Get(key);
            }
            catch (Exception ex)
            {
                logger.Error(ex, string.Empty, key);
            }
            return string.Empty;
        }

        public async Task<string> GetStringAsync(string key)
        {
            if (Client == null)
            {
                return string.Empty;
            }
            try
            {
                return await Client.GetAsync(key);
            }
            catch (Exception ex)
            {
                logger.Error(ex, string.Empty, key);
            }
            return string.Empty;
        }

        /// <summary>
        /// 设置缓存
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="key"></param>
        /// <param name="value"></param>
        /// <param name="overtime">单位-秒</param>
        /// <returns></returns>
        public bool Set(string key, string value, int overtime = -1)
        {
            if (Client == null)
            {
                return false;
            }
            try
            {
                byte[] temparr = SerializeObject(value);
                return Client.Set(key, temparr, overtime);
            }
            catch (Exception ex)
            {
                logger.Error(ex, new { key, value, overtime }, key);
            }
            return false;
        }

        /// <summary>
        /// 设置缓存
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="key"></param>
        /// <param name="value"></param>
        /// <param name="overtime">单位-秒</param>
        /// <returns></returns>
        public async Task<bool> SetAsync<T>(string key, T value, int overtime = -1)
        {
            if (Client == null)
            {
                return false;
            }
            try
            {
                byte[] temparr = SerializeObject(value);
                return await Client.SetAsync(key, temparr, overtime);
            }
            catch (Exception ex)
            {
                logger.Error(ex, new { key, value, overtime }, key);
            }
            return false;
        }

        public async Task<bool> SetStringAsync(string key, string value, int overtime = -1)
        {
            if (Client == null)
            {
                return false;
            }
            try
            {
                return await Client.SetAsync(key, value, overtime);
            }
            catch (Exception ex)
            {
                logger.Error(ex, new { key, value, overtime }, key);
            }
            return false;
        }

        /// <summary>
        /// 如果不存在key则设置缓存
        /// </summary>
        /// <param name="key"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public async Task<bool> SetNxAsync(string key, string value)
        {
            if (Client == null)
            {
                return false;
            }
            try
            {
                return await Client.SetNxAsync(key, value);
            }
            catch (Exception ex)
            {
                logger.Error(ex, new { key, value }, key);
            }
            return false;
        }

        public async Task<long> DelAsync(params string[] key)
        {
            if (Client == null)
            {
                return 0;
            }
            try
            {
                return await Client.DelAsync(key);
            }
            catch (Exception ex)
            {
                logger.Error(ex, new { key });
            }
            return 0;
        }

        public async Task<bool> DelAsync(string key)
        {
            if (Client == null)
            {
                return false;
            }
            try
            {
                return await Client.DelAsync(key) > 0;
            }
            catch (Exception ex)
            {
                logger.Error(ex, new { key });
            }
            return false;
        }

        public long Del(params string[] key)
        {
            if (Client == null)
            {
                return 0;
            }
            try
            {
                return Client.Del(key);
            }
            catch (Exception ex)
            {
                logger.Error(ex, new { key });
            }
            return 0;
        }


        public long DeleteHost(string HostHash, string key)
        {
            if (Client == null)
            {
                return 0;
            }
            try
            {
                return Client.HDel(HostHash, key);
            }
            catch (Exception ex)
            {
                logger.Error(ex, new { key });
            }
            return 0;
        }

        public async Task<bool> ExistsKey(string key)
        {
            if (Client == null)
            {
                return false;
            }
            try
            {
                return await Client.ExistsAsync(key);
            }
            catch (Exception ex)
            {
                logger.Error(ex, new { key }, key);
            }
            return false;
        }

        private byte[] SerializeObject(object obj)
        {
            if (obj == null)
                return null;
            MemoryStream ms = new MemoryStream();
            BinaryFormatter formatter = new BinaryFormatter();
            formatter.Serialize(ms, obj);
            ms.Position = 0;
            byte[] bytes = new byte[ms.Length];
            ms.Read(bytes, 0, bytes.Length);
            ms.Close();
            return bytes;
        }

        private object DeserializeObject(byte[] bytes)
        {
            object obj = null;
            if (bytes == null)
                return obj;
            MemoryStream ms = new MemoryStream(bytes);
            ms.Position = 0;
            BinaryFormatter formatter = new BinaryFormatter();
            obj = formatter.Deserialize(ms);
            ms.Close();
            return obj;
        }

        /// <summary>
        /// 将 key 所储存的值加上给定的增量值（increment）
        /// </summary>
        /// <param name="key"></param>
        /// <param name="value">增量</param>
        /// <returns></returns>
        public async Task<long> IncrByAsync(string key, long value = 1)
        {
            long result = 0;

            try
            {
                if (Client != null)
                {
                    string fillKey = key;
                    return await Client.IncrByAsync(key, value);
                }
            }
            catch (Exception ex)
            {
                logger.Error(ex, new { key, value }, key);
            }
            return result;
        }

        /// <summary>
        ///  向有序集合添加成员，或者更新已存在成员的分数
        /// </summary>
        /// <param name="key"></param>
        /// <param name="score"></param>
        /// <param name="value"></param>
        /// <returns></returns>
        public async Task<long> ZAddAsync(string key, decimal score, string value)
        {
            return await ZAddAsync(key, (score, value));
        }

        /// <summary>
        /// 向有序集合添加一个或多个成员，或者更新已存在成员的分数
        /// </summary>
        /// <param name="key"></param>
        /// <param name="dictionary"></param>
        /// <returns></returns>
        public async Task<long> ZAddAsync(string key, (decimal, object) dictionary)
        {
            return await Client.ZAddAsync(key, dictionary);
        }

        /// <summary>
        /// 通过索引区间返回有序集合成指定区间内的成员
        /// </summary>
        /// <param name="key"></param>
        /// <param name="start"></param>
        /// <param name="stop"></param>
        /// <returns></returns>
        public async Task<IList<string>> ZRangeAsync(string key, long start, long stop)
        {
            return await Client.ZRangeAsync(key, start, stop);
        }
        /// <summary>
        /// 移除集合中一个或多个成员
        /// </summary>
        /// <param name="key"></param>
        /// <param name="members"></param>
        /// <returns></returns>
        public async Task<long> ZRemAsync(string key, params string[] members)
        {
            return await Client.ZRemAsync(key, members);
        }

        /// <summary>
        ///  返回有序集合中指定成员的索引
        /// </summary>
        /// <param name="key"></param>
        /// <param name="member"></param>
        /// <returns></returns>
        public async Task<long?> ZRankAsync(string key, string member)
        {
            return await Client.ZRankAsync(key, member);
        }

        /// <summary>
        /// 返回有序集中，成员的分数值
        /// </summary>
        /// <param name="key"></param>
        /// <param name="member"></param>
        /// <returns></returns>
        public async Task<decimal?> ZScoreAsync(string key, string member)
        {
            return await Client.ZScoreAsync(key, member);
        }

        /// <summary>
        /// 删除并返回有序集合key中的最多count个具有最高得分的成员
        /// </summary>
        /// <param name="key"></param>
        /// <param name="members"></param>
        /// <returns></returns>
        public async Task<List<T>> ZRemAsync<T>(string key, long count)
        {
            var result = await Client.ZPopMaxAsync<T>(key, count);
            return result.Select(x => x.member).ToList();
        }
    }
}
