﻿using AutoMapper;
using Siger.WeComApi.Core.Biz;
using Siger.CommonUtil.Logging;
using System.Collections.Generic;
using System.Linq;
using Siger.ApiTPM.Utilities;
using Microsoft.AspNetCore.Http;
using Senparc.Weixin.Work.Tencent;
using System.Xml.Linq;
using System.IO;
using System;
using Siger.WeComApi.Common.Enums.WeComEnum;
using Newtonsoft.Json.Linq;
using Newtonsoft.Json;
using Siger.WeComApi.Core.Domain.Entity;
using Siger.WeComApi.Core.Repository;
using Siger.WeComApi.Core.Dto.Response;

namespace Siger.WeComApi.Biz.service
{
    public class SigerWeComThirdService : ISigerWeComThirdService
    {
        //企业微信后台开发者设置的token, corpID, EncodingAESKey
        private readonly string Token;
        private readonly string CorpID;
        private readonly string EncodingAESKey;
        //企微应用关联的小程序信息
        private readonly string SuiteID;
        private readonly string SuiteSecret;
        private ISigerLogger<SigerWeComThirdService> _logger;
        private ISigerWeComCompanyRepository _sigerWeComCompanyRepository;

        public SigerWeComThirdService(ISigerLogger<SigerWeComThirdService> logger,
            ISigerWeComCompanyRepository sigerWeComCompanyRepository)
        {
            WeComThirdHelper weComThirdHelper = new WeComThirdHelper();
            Token = weComThirdHelper.GetToken();
            CorpID = weComThirdHelper.GetCorpID();
            EncodingAESKey = weComThirdHelper.GetEncodingAESKey();
            SuiteID = weComThirdHelper.GetSuiteID();
            SuiteSecret = weComThirdHelper.GetSuiteSecret();
            _logger = logger;
            _sigerWeComCompanyRepository = sigerWeComCompanyRepository;
        }

        /// <summary>
        /// 数据回调URL验证
        /// </summary>
        /// <returns></returns>
        public string GetVerify(HttpRequest request)
        {
            string signature = request.Query["msg_signature"];//微信加密签名，msg_signature结合了企业填写的token、请求中的timestamp、nonce参数、加密的消息体
            string timestamp = request.Query["timestamp"];//时间戳
            string nonce = request.Query["nonce"];//随机数

            //在1秒内响应GET请求，响应内容为上一步得到的明文消息内容decryptEchoString（不能加引号，不能带bom头，不能带换行符）
            string echostr = request.Query["echostr"];//加密的随机字符串，以msg_encrypt格式提供。需要解密并返回echostr明文，解密后有random、msg_len、msg、$CorpID四个字段，其中msg即为echostr明文
            if (!string.IsNullOrEmpty(signature) && !string.IsNullOrEmpty(timestamp) && !string.IsNullOrEmpty(nonce) && !string.IsNullOrEmpty(echostr))
            {
                string decryptEchoString = null;
                if (CheckVerifyURL(Token, signature, timestamp, nonce, CorpID, EncodingAESKey, echostr, ref decryptEchoString))
                {
                    if (!string.IsNullOrEmpty(decryptEchoString))
                    {
                        //必须要返回解密之后的明文
                        return decryptEchoString;
                    }
                }
            }
            else
            {
                return "fail";
            }

            return "fail";
        }

        /// <summary>
        /// 验证URL有效性
        /// </summary>
        /// <param name="token">企业微信后台，开发者设置的Token</param>
        /// <param name="signature">签名串，对应URL参数的msg_signature</param>
        /// <param name="timestamp">时间戳</param>
        /// <param name="nonce">随机数</param>
        /// <param name="corpId">ToUserName为企业号的CorpID</param>
        /// <param name="encodingAESKey">企业微信后台，开发者设置的EncodingAESKey</param>
        /// <param name="echostr">随机串，对应URL参数的echostr</param>
        /// <param name="retEchostr">解密之后的echostr，当return返回0时有效</param>
        /// <returns></returns>
        private bool CheckVerifyURL(string token, string signature, string timestamp, string nonce, string corpId, string encodingAESKey, string echostr, ref string retEchostr)
        {
            WXBizMsgCrypt wxcpt = new WXBizMsgCrypt(token, encodingAESKey, corpId);
            int result = wxcpt.VerifyURL(signature, timestamp, nonce, echostr, ref retEchostr);
            if (result != 0)
            {
                return false;//FAIL
            }
            //result==0表示验证成功、retEchostr参数表示明文
            //用户需要将retEchostr作为get请求的返回参数、返回给企业微信号
            return true;
        }

        /// <summary>
        /// 指令回调
        /// </summary>
        /// <returns></returns>
        public string InstructCallback(HttpRequest request) 
        {
            string responseContent = "";
            string postString = GetXMLParameters(request); //xml数据解析
            if (string.IsNullOrEmpty(postString))
            {
                responseContent = "响应失败，未获取到xml中的请求参数";
                return responseContent;
            }
            var xmlDoc = XDocument.Parse(postString);//xml数据转化

            //https://work.weixin.qq.com/api/doc/90001/90143/90613
            //在发生授权、通讯录变更、ticket变化等事件时，企业微信服务器会向应用的“指令回调URL”推送相应的事件消息。
            //消息结构体将使用创建应用时的EncodingAESKey进行加密（特别注意, 在第三方回调事件中使用加解密算法，receiveid的内容为suiteid），请参考接收消息解析数据包。 本章节的回调事件，服务商在收到推送后都必须直接返回字符串 “success”，若返回值不是 “success”，企业微信会把返回内容当作错误信息。
            if (xmlDoc.Root.Element("Encrypt") != null)
            {
                //将post请求的数据进行xml解析，并将<Encrypt> 标签的内容进行解密，解密出来的明文即是用户回复消息的明文
                //接收并读取POST过来的XML文件流
                string decryptionParame = null;  // 解析之后的明文

                // 注意注意:sCorpID
                // @param sReceiveId: 不同场景含义不同，详见文档说明（[消息加密时为 CorpId]ToUserName：企业微信的CorpID，当为第三方应用回调事件时，CorpID的内容为suiteid）
                string signature = request.Query["msg_signature"];//微信加密签名，msg_signature结合了企业填写的token、请求中的timestamp、nonce参数、加密的消息体
                string timestamp = request.Query["timestamp"];//时间戳
                string nonce = request.Query["nonce"];//随机数
                
                WXBizMsgCrypt crypt = new WXBizMsgCrypt(Token, EncodingAESKey, xmlDoc.Root.Element("ToUserName").Value);
                var result = crypt.DecryptMsg(signature, timestamp, nonce, postString, ref decryptionParame);
                if (result != 0)
                {
                    return "fail";
                }

                //解密后的数据 囊括了身份信息 验证信息
                WeComThirdHelper weComThirdHelper = new WeComThirdHelper();
                var decryptInfo = XDocument.Parse(decryptionParame);
                var infoType = decryptInfo.Root.Element("InfoType");
                if (infoType != null)
                {
                    switch (infoType.Value)
                    {
                        case "suite_ticket":
                            string suitTicket = decryptInfo.Root.Element("SuiteTicket").Value;
                            weComThirdHelper.SetSuitTicket(suitTicket);
                            break;
                        case "create_auth":
                            //从企业微信应用市场发起授权时，企业微信后台会推送授权成功通知
                            //获取auth_code 临时授权码
                            string authCode = decryptInfo.Root.Element("AuthCode").Value;
                            GetPermentCode(authCode);
                            break;
                        case "change_auth":
                            break;
                        case "cancel_auth":
                            //获取corp_id
                            break;
                        case "register_corp":
                            break;
                        case "batch_job_result":
                            //通讯录id转译异步任务回调  https://open.work.weixin.qq.com/api/doc/90001/90143/91875
                            break;
                        default:
                            _logger.Info("infoType：" + infoType);
                            break;
                    }

                }
                //响应应答处理
                return ReceiveResponse(decryptionParame, timestamp, signature, Token, EncodingAESKey, CorpID);
            }
            return "success";
        }

        public void GetPermentCode(string authCode)
        {
            //通过auth code获取公司信息及永久授权码
            WeComThirdHelper weComThirdHelper = new WeComThirdHelper();
            JObject postJson = new JObject();
            postJson["auth_code"] = authCode;
            var suitAccessToken = weComThirdHelper.GetSuiteAccessToken();
            string pUrl = weComThirdHelper.GetPermanentCodeUrl() + suitAccessToken;
            string corpInfo = weComThirdHelper.post(pUrl + "", postJson);
            //将授权的客户企业存入数据库   补充报错的判断！！！
            InsertWeComCompany(corpInfo);
        }

        #region 响应应答处理
        /// <summary>
        /// 响应应答处理
        /// </summary>
        /// <param name="sMsg">解密参数</param>
        /// <param name="timestamp">时间戳</param>
        /// <param name="signature">签名</param>
        /// <param name="sToken">企业微信后台，开发者设置的Token</param>
        /// <param name="sEncodingAESKey">开发者设置的EncodingAESKey</param>
        /// <param name="sCorpID">业号corpid是企业号的专属编号（CorpID）</param>
        /// <returns></returns>
        public string ReceiveResponse(string sMsg, string timestamp, string signature, string sToken, string sEncodingAESKey, string sCorpID)
        {
            string responseMessage = "success";//响应内容   
            var xmlDoc = XDocument.Parse(sMsg);//xml数据转化

            //区分普通消息与第三方应用授权推送消息，MsgType有值说明是普通消息，反之则是第三方应用授权推送消息
            if (xmlDoc.Root.Element("MsgType") != null)
            {
                var msgType = (ResponseMsgType)Enum.Parse(typeof(ResponseMsgType), xmlDoc.Root.Element("MsgType").Value, true);
                switch (msgType)
                {
                    case ResponseMsgType.Text://文本消息
                        responseMessage = ResponseMessageText(xmlDoc, timestamp, signature, sToken, sEncodingAESKey, sCorpID);
                        break;
                    case ResponseMsgType.Image:
                        responseMessage = ResponseMessageImage();
                        break;
                    case ResponseMsgType.Voice:
                        responseMessage = ResponseMessageVoice();
                        break;
                    case ResponseMsgType.Video:
                        responseMessage = ResponseMessageVideo();
                        break;
                    case ResponseMsgType.News:
                        responseMessage = ResponseMessageNews();
                        break;
                }
            }
            else if (xmlDoc.Root.Element("InfoType") != null)
            {
                //第三方回调
                var infoType = (ResponseInfoType)Enum.Parse(typeof(ResponseInfoType), xmlDoc.Root.Element("InfoType").Value, true);

                switch (infoType)
                {
                    case ResponseInfoType.suite_ticket:
                        {
                            //LoggerHelper._.Warn("suite_ticket===>>>>>,进来了，获取到的SuiteTicket票据为" + xmlDoc.Root.Element("SuiteTicket").Value);
                        }
                        break;
                }
            }
            else
            {
                //其他情况
            }

            // result==0表示解密成功，sMsg表示解密之后的明文xml串
            //服务器未正确返回响应字符串 “success”
            return responseMessage;
        }

        /// <summary>
        /// 消息文本回复
        /// </summary>
        /// <returns></returns>
        public string ResponseMessageText(XDocument xmlDoc, string timestamp, string nonce, string sToken, string sEncodingAESKey, string sCorpID)
        {
            string FromUserName = xmlDoc.Root.Element("FromUserName").Value;
            string ToUserName = xmlDoc.Root.Element("ToUserName").Value;
            string Content = xmlDoc.Root.Element("Content").Value;

            string xml = "<xml>";
            xml += "<ToUserName><![CDATA[" + ToUserName + "]]></ToUserName>";
            xml += "<FromUserName><![CDATA[" + FromUserName + "]]></FromUserName>";
            xml += "<CreateTime>" + GetCurrentTimeUnix() + "</CreateTime>";
            xml += "<MsgType><![CDATA[text]]></MsgType>";
            xml += "<Content><![CDATA[" + Content + "]]></Content>";
            xml += "</xml>";
            //"" + Content + "0";//回复内容 FuncFlag设置为1的时候，自动星标刚才接收到的消息，适合活动统计使用
            WXBizMsgCrypt wxcpt = new WXBizMsgCrypt(sToken, sEncodingAESKey, sCorpID);
            string sEncryptMsg = "";// 加密后的可以直接回复用户的密文;
            wxcpt.EncryptMsg(xml, timestamp, nonce, ref sEncryptMsg);

            //返回
            return sEncryptMsg;
        }

        /// <summary>
        /// 图片消息
        /// </summary>
        /// <returns></returns>

        public string ResponseMessageImage()
        {
            return "success";
        }

        /// <summary>
        /// 语音消息
        /// </summary>
        /// <returns></returns>
        public string ResponseMessageVoice()
        {
            return "success";
        }

        /// <summary>
        /// 视频消息
        /// </summary>
        /// <returns></returns>
        public string ResponseMessageVideo()
        {
            return "success";
        }

        /// <summary>
        /// 图文消息
        /// </summary>
        /// <returns></returns>
        public string ResponseMessageNews()
        {
            return "success";
        }

        #endregion

        /// <summary>
        /// 获取当前时间戳
        /// </summary>
        /// <returns></returns>
        public static string GetCurrentTimeUnix()
        {
            TimeSpan cha = (DateTime.Now - TimeZone.CurrentTimeZone.ToLocalTime(new System.DateTime(1970, 1, 1)));
            long t = (long)cha.TotalSeconds;
            return t.ToString();
        }

        /// <summary>
        /// 解析Request.Body中的参数
        /// </summary>
        /// <returns></returns>
        private string GetXMLParameters(HttpRequest request)
        {
            string replyMsg = "";
            request.EnableBuffering();
            long? length = request.ContentLength;
            if (length != null && length > 0)
            {
                StreamReader streamReader = new StreamReader(request.Body);
                replyMsg = streamReader.ReadToEndAsync().Result;
            }
            request.Body.Position = 0;
            return replyMsg;
        }

        private string InsertWeComCompany(string corpInfo) 
        {
            JObject corpJson = (JObject)JsonConvert.DeserializeObject(corpInfo);
            if (corpJson["errcode"] != null)
            {
                string errorMsg = string.Format("获取企业信息及永久授权码失败:errcode{0},errmsg{1}", corpJson["errcode"], corpJson["errmsg"]);
                throw new Exception(errorMsg);
            }
            var authCorpInfo = corpJson["auth_corp_info"].ToString();
            siger_wecom_company company = JsonConvert.DeserializeObject<siger_wecom_company>(authCorpInfo);
            //赋值永久授权码permanent_code
            var permanentCode = corpJson["permanent_code"].ToString();
            company.permanent_code = permanentCode;
            return _sigerWeComCompanyRepository.AddSigerWeComCompany(company);

        }

        public ResponseWeComUser GetUserInfo(string code)
        {
            //获取访客身份
            WeComThirdHelper weComThirdHelper = new WeComThirdHelper();
            JObject postJson = new JObject();
            var suite_access_token = weComThirdHelper.GetSuiteAccessToken();
            var userinfo3rdUrl = weComThirdHelper.GetUserinfo3rdUrl() + "?suite_access_token=" + suite_access_token + "&js_code=" + code + "&grant_type=" + "authorization_code";
            string response = weComThirdHelper.post(userinfo3rdUrl, null);
            JObject userInfoJson = (JObject)JsonConvert.DeserializeObject(response);
            if (userInfoJson["errcode"].ToString() != "0")
            {
                string errorMsg = string.Format("获取访客身份失败,errcode: {0},errmsg: {1}", userInfoJson["errcode"], userInfoJson["errmsg"]);
                throw new Exception(errorMsg);
            }

            //获取当前用户所属公司的永久授权码
            siger_wecom_company curCompany = _sigerWeComCompanyRepository.GetSigerWeComCompany("", userInfoJson["corpid"].ToString());
            if (curCompany == null) 
            {
                string errorMsg = string.Format("获取访客身份失败,errcode: {0},errmsg: {1}", "", "数据库中未找到当前授权公司");
                throw new Exception(errorMsg);
            }

            //获取客户企业凭证
            string accessToken = weComThirdHelper.GetCorpAccessToken(curCompany.corpid, curCompany.permanent_code);

            //获取访客敏感信息
            var userAllInfo = weComThirdHelper.GetUserAllInfo(accessToken, userInfoJson["userid"].ToString());
            JObject userAllInfoJson = (JObject)JsonConvert.DeserializeObject(userAllInfo);
            ResponseWeComUser company = JsonConvert.DeserializeObject<ResponseWeComUser>(userAllInfoJson.ToString());
            company.corpid = curCompany.corpid;
            company.corpname = curCompany.corp_name;

            return company;
        }

    }
}
