﻿using Microsoft.EntityFrameworkCore;
using Siger.WeComApi.Core.Domain;
using Siger.WeComApi.Core.Domain.Paged;
using Siger.WeComApi.Core.Repository;
using Siger.WeComApi.Repository.Extensions;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Threading.Tasks;

namespace Siger.WeComApi.Repository.Repositories
{
    /// <summary>
    /// 仓储基类
    /// 本基类实现的读写分离，需要读写库数据，需要手动实现
    /// </summary>
    /// <typeparam name="TEntity"></typeparam>
    internal class RepositoryBase<TEntity> : IRepositoryBase<TEntity> where TEntity : EntityBase
    {
        /// <summary>
        /// 主库,写操作
        /// </summary>
        protected readonly BaseDataDbContext DbMaster;

        /// <summary>
        /// 从库,读操作
        /// </summary>
        protected readonly BaseDataReadDbContext DbSlave;
        public RepositoryBase(BaseDataDbContext dbMaster, BaseDataReadDbContext dbSlave)
        {
            this.DbMaster = dbMaster;
            this.DbSlave = dbSlave;
        }

        /// <summary>
        /// Insert
        /// </summary>
        /// <param name="entity"></param>
        public void Insert(TEntity entity)
        {
            DbMaster.Set<TEntity>().Add(entity);
        }

        /// <summary>
        /// insert
        /// </summary>
        /// <param name="entity"></param>
        /// <returns></returns>
        public async Task InsertAsync(TEntity entity)
        {
            await DbMaster.Set<TEntity>().AddAsync(entity);
        }

        /// <summary>
        /// 批量 insert
        /// </summary>
        /// <param name="entities"></param>
        /// <returns></returns>
        public async Task InsertAsync(IEnumerable<TEntity> entities)
        {
            await DbMaster.Set<TEntity>().AddRangeAsync(entities);
        }

        /// <summary>
        /// Update
        /// </summary>
        /// <param name="entity"></param>
        public void Update(TEntity entity)
        {
            DbMaster.Set<TEntity>().Update(entity);
        }

        /// <summary>
        /// UpdateRange
        /// </summary>
        /// <param name="entity"></param>
        public void UpdateRange(params TEntity[] entities)
        {
            DbMaster.Set<TEntity>().UpdateRange(entities);
        }


        public void Delete(object id)
        {
            var entity = DbSlave.Set<TEntity>().Find(id);
            if (entity != null)
            {
                Delete(entity);
            }
        }

        public async Task DeleteAsync(object id)
        {
            await Task.Run(() =>
            {
                var entity = DbSlave.Set<TEntity>().Find(id);
                if (entity != null)
                {
                    Delete(entity);
                }
            });
        }

        public void Delete(TEntity entity)
        {
            DbMaster.Remove(entity);
        }

        public async Task DeleteAsync(TEntity entity)
        {
            await Task.Run(() =>
            {
                Delete(entity);
            });
        }

        /// <summary>
        /// 根据主键获取对象
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        public TEntity Get(object key)
        {
            return DbSlave.Set<TEntity>().Find(key);
        }

        /// <summary>
        /// 根据主键获取对象
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        public async Task<TEntity> GetAsync(object key)
        {
            return await DbSlave.Set<TEntity>().FindAsync(key);
        }

        /// <summary>
        /// 根据主键获取对象
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        public TEntity Get(Expression<Func<TEntity, bool>> predicate)
        {
            return DbSlave.Set<TEntity>().Where(predicate).FirstOrDefault();
        }

        /// <summary>
        /// 根据条件获取单个对象
        /// </summary>
        /// <param name="predicate"></param>
        /// <returns></returns>
        public async Task<TEntity> GetAsync(Expression<Func<TEntity, bool>> predicate)
        {
            try
            {
                predicate ??= (x => true);
                return await DbSlave.Set<TEntity>().Where(predicate).FirstOrDefaultAsync();
            }
            catch (Exception ex)
            {
                throw ex;
            }

        }

        /// <summary>
        /// 排序后取第一条
        /// </summary>
        /// <param name="predicate"></param>
        /// <param name="orderByPropertyName"></param>
        /// <param name="desc"></param>
        /// <returns></returns>
        public async Task<TEntity> GetAsync(Expression<Func<TEntity, bool>> predicate, string orderByPropertyName = null, bool desc = true)
        {
            predicate ??= (x => true);
            TEntity entity;
            if (!string.IsNullOrEmpty(orderByPropertyName))
            {
                entity = await DbSlave.Set<TEntity>().Where(predicate).OrderByPropertyName(orderByPropertyName, desc).FirstOrDefaultAsync();
            }
            else
            {
                entity = await DbSlave.Set<TEntity>().Where(predicate).FirstOrDefaultAsync();
            }
            return entity;
        }

        /// <summary>
        /// 是否存在满足条件的数据
        /// </summary>
        /// <param name="predicate"></param>
        /// <returns></returns>
        public async Task<bool> IsExistAsync(Expression<Func<TEntity, bool>> predicate)
        {
            var entity = await GetAsync(predicate);
            return entity != null;
        }

        public List<TEntity> GetList(Expression<Func<TEntity, bool>> predicate = null, string orderByPropertyName = null, bool desc = true)
        {
            predicate ??= (x => true);
            if (!string.IsNullOrEmpty(orderByPropertyName))
            {
                return DbSlave.Set<TEntity>().Where(predicate).AsNoTracking().OrderByPropertyName(orderByPropertyName, desc).ToList();
            }
            return DbSlave.Set<TEntity>().Where(predicate).AsNoTracking().ToList();
        }

        /// <summary>
        /// 集合查询
        /// </summary>
        /// <param name="predicate"></param>
        /// <param name="orderByPropertyName"></param>
        /// <param name="desc"></param>
        /// <returns></returns>
        public async Task<List<TEntity>> GetListAsync(Expression<Func<TEntity, bool>> predicate = null, string orderByPropertyName = null, bool desc = true)
        {

            predicate ??= (x => true);
            if (!string.IsNullOrEmpty(orderByPropertyName))
            {
                return await DbSlave.Set<TEntity>().Where(predicate).AsNoTracking().OrderByPropertyName(orderByPropertyName, desc).ToListAsync();
            }
            return await DbSlave.Set<TEntity>().Where(predicate).AsNoTracking().ToListAsync();
        }

        /// <summary>
        /// 分页查询
        /// </summary>
        /// <param name="pageIndex"></param>
        /// <param name="pageSize"></param>
        /// <param name="predicate"></param>
        /// <param name="orderByPropertyName"></param>
        /// <param name="desc"></param>
        /// <returns></returns>
        public async Task<PagedCollectionResult<TEntity>> GetPagedListAsync(int pageIndex, int pageSize, Expression<Func<TEntity, bool>> predicate = null, string orderByPropertyName = null, bool desc = true)
        {
            long totalCount = 0;
            IList<TEntity> entities = null;
            predicate ??= (x => true);
            totalCount = await DbSlave.Set<TEntity>().CountAsync(predicate);
            if (!string.IsNullOrEmpty(orderByPropertyName))
            {
                entities = await DbSlave.Set<TEntity>().Where(predicate).OrderByPropertyName(orderByPropertyName, desc)
                                 .Skip((pageIndex - 1) * pageSize).Take(pageSize)
                                 .AsNoTracking().ToListAsync();
            }
            else
            {
                entities = await DbSlave.Set<TEntity>().Where(predicate)
                                 .Skip((pageIndex - 1) * pageSize).Take(pageSize)
                                 .AsNoTracking().ToListAsync();
            }
            return new PagedCollectionResult<TEntity>(entities, totalCount);
        }

        /// <summary>
        /// sql 执行
        /// </summary>
        /// <param name="sql"></param>
        /// <returns></returns>
        public async Task<int> ExecuteSqlRawAsync(string sql)
        {
            return await DbMaster.Database.ExecuteSqlRawAsync(sql);
        }

    }
}
