﻿using Siger.CommonUtil.Extensions;
using Siger.CommonUtil.Logging;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Runtime.CompilerServices;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Siger.WeComApi.Common.HttpClients
{
    /// <summary>
    /// HttpClient  基础类 
    /// 长连接继承使用
    /// </summary>
    /// <typeparam name="TCategoryName"></typeparam>
    public abstract class HttpClientBase<TCategoryName> where TCategoryName : class
    {
        /// <summary>
        /// 默认超时时间
        /// </summary>
        protected const int DefaultMillisecondsDelay = 3000;

        protected ISigerLogger<TCategoryName> logger { get; }
        protected IHttpClientFactory httpClientFactory { get; set; }

        private readonly HttpClient httpClient;
        public HttpClientBase(
            ISigerLogger<TCategoryName> logger,
            IHttpClientFactory httpClientFactory)
        {
            this.logger = logger ?? throw new ArgumentNullException(nameof(logger));
            this.httpClientFactory = httpClientFactory ?? throw new ArgumentNullException(nameof(httpClientFactory));

            httpClient = InitHttpClient() ?? throw new WebException("HttpClient Uninitialized");
        }

        /// <summary>
        /// 初始化
        /// </summary>
        /// <returns></returns>
        protected virtual HttpClient InitHttpClient()
        {
            var httpClient = httpClientFactory.CreateClient();
            httpClient.BaseAddress = new Uri(GetBaseUrl());
            httpClient.DefaultRequestHeaders.Connection.Add("keep-alive");
            httpClient.Timeout = TimeSpan.FromMilliseconds(DefaultMillisecondsDelay);
            //http warm up
            var headRequest = new HttpRequestMessage(HttpMethod.Head, string.Empty);
            httpClient.SendAsync(headRequest).Wait();
            return httpClient;
        }

        /// <summary>
        /// 获取基础url
        /// </summary>
        /// <returns></returns>
        protected abstract string GetBaseUrl();

        /// <summary>
        ///   Send a GET request to the specified Uri with a cancellation token as an asynchronous
        /// </summary>
        /// <param name="url"></param>
        /// <param name="encoding"></param>
        /// <param name="filter1"></param>
        /// <param name="filter2"></param>
        /// <param name="callerName"></param>
        /// <param name="millisecondsDelay"></param>
        /// <returns></returns>
        public async Task<string> GetAsync(string url, Encoding encoding = null, string filter1 = "", string filter2 = "", string callerName = "", int millisecondsDelay = DefaultMillisecondsDelay)
        {
            Dictionary<string, dynamic> extraInfo = new Dictionary<string, dynamic>();
            extraInfo.Add("url", $"{httpClient.BaseAddress.AbsoluteUri}{url}");
            extraInfo.Add("method", "get");
            Stopwatch stopwatch = Stopwatch.StartNew();
            string responseText = string.Empty;
            try
            {
                if (url.StartsWith(AppConstants.Protocols.HTTPS, StringComparison.CurrentCultureIgnoreCase))
                {
                    ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls;
                }
                var cts = new CancellationTokenSource((millisecondsDelay <= 0 ? DefaultMillisecondsDelay : millisecondsDelay));
                var response = await httpClient.GetAsync(url, cts.Token);
                var resultBytes = response.IsSuccessStatusCode ? await response.Content.ReadAsByteArrayAsync() : Array.Empty<byte>();

                responseText = (encoding ?? Encoding.UTF8).GetString(resultBytes);
                stopwatch.Stop();
                extraInfo.Add("Elapsed", stopwatch.ElapsedMilliseconds.ToString());
                extraInfo.Add("responseText", responseText);
                logger.Info(extraInfo, filter1, filter2, callerName);
            }
            catch (Exception ex)
            {
                stopwatch.Stop();
                extraInfo.Add("Elapsed", stopwatch.ElapsedMilliseconds.ToString());
                extraInfo.Add("ex", ex);
                logger.Error(ex, extraInfo, filter1, filter2, callerName);
                throw;
            }
            return responseText;
        }
       
        /// <summary>
        /// get请求 序列化结果
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="url"></param>
        /// <param name="encoding"></param>
        /// <param name="filter1"></param>
        /// <param name="filter2"></param>
        /// <param name="millisecondsDelay"></param>
        /// <param name="callerName"></param>
        /// <returns></returns>
        public async Task<T> GetAsync<T>(string url, Encoding encoding = null, string filter1 = "", string filter2 = "", int millisecondsDelay = DefaultMillisecondsDelay, [CallerMemberName] string callerName = "")
        {
            var response = await GetAsync(url, encoding, filter1, filter2, callerName, millisecondsDelay);
            return response.IsNullOrWhiteSpace() ? default(T) : response.FromJson<T>();
        }
        
        /// <summary>
        /// post请求 返回String
        /// </summary>
        /// <param name="url"></param>
        /// <param name="postData"></param>
        /// <param name="headers"></param>
        /// <param name="contentType"></param>
        /// <param name="encoding"></param>
        /// <param name="filter1"></param>
        /// <param name="filter2"></param>
        /// <param name="callerName"></param>
        /// <param name="millisecondsDelay"></param>
        /// <returns></returns>
        public virtual async Task<string> PostAsync(string url, string postData, IDictionary<string, string> headers = null, string contentType = null, Encoding encoding = null, string filter1 = "", string filter2 = "", string callerName = "", int millisecondsDelay = DefaultMillisecondsDelay)
        {
            Dictionary<string, dynamic> extraInfo = new Dictionary<string, dynamic>();
            extraInfo.Add("url", $"{httpClient.BaseAddress.AbsoluteUri}{url}");
            extraInfo.Add("method", "post");
            extraInfo.Add("requestText", postData);
            Stopwatch stopwatch = Stopwatch.StartNew();
            string responseText = string.Empty;
            try
            {
                if (url.StartsWith(AppConstants.Protocols.HTTPS, StringComparison.CurrentCultureIgnoreCase))
                {
                    ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls;
                }
                using (var requestMessage = new HttpRequestMessage(HttpMethod.Post, $"{httpClient.BaseAddress.AbsoluteUri}{url}"))
                using (var cts = new CancellationTokenSource((millisecondsDelay <= 0 ? DefaultMillisecondsDelay : millisecondsDelay)))
                {
                    requestMessage.Content = new StringContent(postData);
                    requestMessage.Content.Headers.ContentType = new MediaTypeHeaderValue(contentType.NullOrEmpty2Default(AppConstants.HttpMediaTypes.ApplicationJson));

                    if (headers?.Count > 0)
                    {
                        foreach (var item in headers)
                        {
                            requestMessage.Headers.Add(item.Key, item.Value);
                        }
                    }
                    var response = await httpClient.SendAsync(requestMessage, cts.Token);
                    var resultBytes = response.IsSuccessStatusCode ? await response.Content.ReadAsByteArrayAsync() : Array.Empty<byte>();
                    responseText = (encoding ?? Encoding.UTF8).GetString(resultBytes);
                }
                stopwatch.Stop();
                extraInfo.Add("Elapsed", stopwatch.ElapsedMilliseconds.ToString());
                extraInfo.Add("responseText", responseText);
                logger.Info(extraInfo, filter1, filter2, callerName);
            }
            catch (Exception ex)
            {
                stopwatch.Stop();
                extraInfo.Add("Elapsed", stopwatch.ElapsedMilliseconds.ToString());
                extraInfo.Add("ex", ex);
                logger.Error(ex, extraInfo, filter1, filter2, callerName);

                throw;
            }

            return responseText;
        }
       
        /// <summary>
        /// Post请求 序列化结果
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="url"></param>
        /// <param name="postData"></param>
        /// <param name="headers"></param>
        /// <param name="encoding"></param>
        /// <param name="filter1"></param>
        /// <param name="filter2"></param>
        /// <param name="millisecondsDelay"></param>
        /// <param name="callerName"></param>
        /// <returns></returns>
        public virtual async Task<T> PostAsync<T>(string url, string postData, IDictionary<string, string> headers = null, Encoding encoding = null, string filter1 = "", string filter2 = "", int millisecondsDelay = DefaultMillisecondsDelay, [CallerMemberName] string callerName = "")
        {
            var response = await PostAsync(url, postData, headers, null, encoding, filter1, filter2, callerName, millisecondsDelay);
            return response.IsNullOrWhiteSpace() ? default(T) : response.FromJson<T>();
        }

        /// <summary>
        ///  Post请求 序列化结果,结构化参数
        /// </summary>
        /// <typeparam name="TIn"></typeparam>
        /// <typeparam name="TOut"></typeparam>
        /// <param name="url"></param>
        /// <param name="postData"></param>
        /// <param name="headers"></param>
        /// <param name="encoding"></param>
        /// <param name="filter1"></param>
        /// <param name="filter2"></param>
        /// <param name="millisecondsDelay"></param>
        /// <param name="callerName"></param>
        /// <returns></returns>
        public virtual async Task<TOut> PostAsync<TIn, TOut>(string url, TIn postData, IDictionary<string, string> headers = null, Encoding encoding = null, string filter1 = "", string filter2 = "", int millisecondsDelay = DefaultMillisecondsDelay, [CallerMemberName] string callerName = "")
            where TOut : class, new()
        {
            var response = await PostAsync(url, postData.ToJson(), headers, null, encoding, filter1, filter2, callerName, millisecondsDelay);
            return response.IsNullOrWhiteSpace() ? default(TOut) : response.FromJson<TOut>();
        }
        
       /// <summary>
       /// Post 请求 返回xml结构
       /// </summary>
       /// <typeparam name="T"></typeparam>
       /// <param name="url"></param>
       /// <param name="xmlString"></param>
       /// <param name="headers"></param>
       /// <param name="encoding"></param>
       /// <param name="filter1"></param>
       /// <param name="filter2"></param>
       /// <param name="millisecondsDelay"></param>
       /// <param name="callerName"></param>
       /// <returns></returns>
        public virtual async Task<T> PostWithXmlResponseAsync<T>(string url, string xmlString, IDictionary<string, string> headers = null, Encoding encoding = null, string filter1 = "", string filter2 = "", int millisecondsDelay = DefaultMillisecondsDelay, [CallerMemberName] string callerName = "")
            where T : class, new()
        {
            var response = await PostAsync(url, xmlString, headers, null, encoding, filter1, filter2, callerName, millisecondsDelay);
            return response.IsNullOrWhiteSpace() ? default(T) : response.FromXml<T>();
        }
    }
}
