using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http.Features;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Server.Kestrel.Core;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Hosting;
using Newtonsoft.Json.Converters;
using Newtonsoft.Json.Serialization;
using Siger.CommonUtil.Logging;
using Siger.CommonUtil.Logging.OperateLog;
using Siger.WeComApi.Common;
using Siger.WeComApi.Common.config;
using Siger.WeComApi.Common.Filter;
using Siger.WeComApi.Common.Helpers;
using Siger.WeComApi.Common.HttpClients;
using Siger.WeComApi.Common.Logger;
using Siger.WeComApi.Repository;
using Siger.SysLogApi.Common.AutoMap;

namespace Siger.WeComApi.WebApi
{
    public class Startup
    {
        private string _corsKeyName = "Cors";
        public Startup(IConfiguration configuration)
        {
            Configuration = configuration;
        }

        public IConfiguration Configuration { get; }

        // This method gets called by the runtime. Use this method to add services to the container.
        public void ConfigureServices(IServiceCollection services)
        {
            services.AddCors(options =>
            {
                options.AddPolicy(_corsKeyName, builder => builder.AllowAnyOrigin()
                .AllowAnyHeader().AllowAnyMethod());
            });
            services.AddMvc(option =>
            {
                option.EnableEndpointRouting = false;
                //option.Filters.Add<ModelValidateFilter>();
                option.Filters.Add<ApiExceptionFilter>();
                option.Filters.Add<ResultFilter>();
                option.Filters.Add<TokenValidateFilter>();
            })
                .SetCompatibilityVersion(CompatibilityVersion.Version_3_0);
            services.AddControllers().AddNewtonsoftJson(option =>
            {
                option.SerializerSettings.ContractResolver = new CamelCasePropertyNamesContractResolver();
                option.SerializerSettings.Converters.Add(new IsoDateTimeConverter() { DateTimeFormat = "yyyy-MM-dd HH:mm:ss" });
            });
            // If using Kestrel:
            services.Configure<KestrelServerOptions>(options =>
            {
                options.AllowSynchronousIO = true;
            });

            // If using IIS:
            services.Configure<IISServerOptions>(options =>
            {
                options.AllowSynchronousIO = true;
            });
            //services.AddDbContext<TpmDbContext>();
            services.RegisterConfigCenter(Configuration);
            services.AddSigerLogger();
            services.AutoMapperConfigure();
            services.AddHostedService<LogBackgroundService>();
            services.RegisterService();
            services.AddSwagger();
            services.Configure<FormOptions>(x =>
            {
                x.ValueLengthLimit = int.MaxValue;
                x.MultipartBodyLengthLimit = int.MaxValue; // In case of multipart
            });
            services.AddDbContext<BaseDataReadDbContext>();
            services.AddDbContext<BaseDataDbContext>();
            services.AddCustomHttpClient();
            var serviceProvider = services.BuildServiceProvider();
            ServiceContainer.SetServiceProvider(serviceProvider);
            SigerLogger.Info($"{AppConstants.AppID} has been activated");
            var startTask = ConfigManager.GetValue("Task", "Start", 0);
            if (startTask == 1 && Configuration["IsRunTask"] != null)
            {

                //Scheduler.StartUp();
            }
        }

        public void Configure(IApplicationBuilder app, IWebHostEnvironment env)
        {
            if (env.IsDevelopment())
            {
                app.UseDeveloperExceptionPage();
            }
            app.RegisterSwagger();
            app.UseHttpsRedirection();

            app.UseRouting();
            app.UseCors(_corsKeyName);
            app.UseMvc();
            app.UseEndpoints(endpoints =>
            {
                endpoints.MapControllers();
            });
            app.UseHttpsRedirection();
            app.UseStaticFiles();
        }
    }
}
