﻿using Microsoft.Extensions.DependencyInjection;
using Siger.CommonUtil.Extensions;
using Siger.CommonUtil.Logging;
using Siger.WeComApi.Common.cache;
using Siger.WeComApi.Common.CommonModels;
using System;
using System.Diagnostics;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace Siger.WeComApi.Common.Helpers
{
    /// <summary>
    /// 登录Token帮助
    /// </summary>
    public class TokenHelper
    {
        public const string Prefix = "Token_";
        private readonly ISigerLogger<TokenHelper> logger;

        private static TokenHelper tokenHelper;

        /// <summary>
        /// 单例实现
        /// </summary>
        public static TokenHelper Instance
        {
            get
            {
                if (tokenHelper == null)
                {
                    tokenHelper = new TokenHelper();
                }
                return tokenHelper;
            }
        }
        private readonly int tokenExpiryMinutes;

        private TokenHelper()
        {
            logger = ServiceContainer.Current.GetService<ISigerLogger<TokenHelper>>();
            tokenExpiryMinutes = ConfigManager.GetValue("RedisSetting", "TokenExpiryMinutes", 750);
        }


        /// <summary>
        /// 不为空且已经登录
        /// </summary>
        /// <param name="token"></param>
        /// <returns></returns>
        public bool IsValidToken(string token)
        {
            try
            {
                var value = CacheUtility.Instance.GetString(token);
                if (!value.IsNullOrWhiteSpace())
                {
                    if (!value.StartsWith(Prefix))
                    {
                        return true;
                    }
                }
            }
            catch (Exception ex)
            {
                logger.Error(ex, $"IsValidToken by {token} failed", token);
                return false;
            }
            return false;
        }

        public async Task ResetToken(string token)
        {
            var value = await CacheUtility.Instance.GetStringAsync(token);
            if (!value.IsNullOrWhiteSpace())
            {
                await CacheUtility.Instance.SetAsync(token, value, (int)TimeSpan.FromMinutes(tokenExpiryMinutes).TotalSeconds);
            }
        }

        public TokenValue GetTokenValue(string token)
        {
            try
            {
                var stopwatch = new Stopwatch();
                stopwatch.Start();
                var value = CacheUtility.Instance.GetString(token);
                if (!value.IsNullOrWhiteSpace())
                {
                    var values = value.Split('_');
                    if (values.Length >= 5)
                    {
                        int.TryParse(values[0], out var companyid);
                        int.TryParse(values[1], out var projectid);
                        int.TryParse(values[2], out var userid);
                        int.TryParse(values[3], out var roleid);
                        int.TryParse(values[4], out var companytype);
                        var tokenValue = new TokenValue(companyid, projectid, userid, roleid, companytype);
                        return tokenValue;
                    }
                }
                stopwatch.Stop();
                logger.Info(new { timespan = stopwatch.ElapsedMilliseconds }, token);
            }
            catch (Exception ex)
            {
                logger.Error(ex, $"GetTokenValue by {token} failed", token);
                return null;
            }
            return null;
        }

        public static string GetQueryString(string name, string url)
        {
            Regex re = new Regex(@"(^|&)?(\w+)=([^&]+)(&|$)?", RegexOptions.Compiled);
            MatchCollection mc = re.Matches(url);
            foreach (Match m in mc)
            {
                if (m.Result("$2").Equals(name))
                {
                    return m.Result("$3");
                }
            }
            return "";
        }
    }
}
