﻿using Siger.CommonUtil.Extensions;
using Siger.CommonUtil.Logging;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Net;
using System.Net.Http;
using System.Runtime.CompilerServices;
using System.Text;
using System.Threading.Tasks;

namespace Siger.WeComApi.Common.HttpClients
{
    public class GenericHttpClient<TCategoryName> where TCategoryName : class
    {
        private readonly ISigerLogger<TCategoryName> logger;
        private readonly IHttpClientFactory httpClientFactory;

        public GenericHttpClient(ISigerLogger<TCategoryName> logger, IHttpClientFactory httpClientFactory)
        {
            this.logger = logger ?? throw new ArgumentNullException(nameof(logger));
            this.httpClientFactory = httpClientFactory ?? throw new ArgumentNullException(nameof(httpClientFactory));
        }

        #region async
        public async Task<string> GetAsync(string url, int timeout = 6000, Encoding encoding = null, string filter1 = "", string filter2 = "", string callerName = "")
        {
            Dictionary<string, dynamic> extraInfo = new Dictionary<string, dynamic>();
            extraInfo.Add("url", url);
            extraInfo.Add("method", "get");
            Stopwatch stopwatch = Stopwatch.StartNew();
            string responseText = string.Empty;

            try
            {
                var client = httpClientFactory.CreateClient();
                if (url.StartsWith(AppConstants.Protocols.HTTPS, StringComparison.CurrentCultureIgnoreCase))
                {
                    ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls;
                }
                if (timeout > 0)
                {
                    client.Timeout = TimeSpan.FromMilliseconds(timeout);
                }
                var response = await client.GetAsync(url);
                var resultBytes = response.IsSuccessStatusCode ? await response.Content.ReadAsByteArrayAsync() : Array.Empty<byte>();

                responseText = (encoding ?? Encoding.UTF8).GetString(resultBytes);
                stopwatch.Stop();
                extraInfo.Add("Elapsed", stopwatch.ElapsedMilliseconds.ToString());
                extraInfo.Add("responseText", responseText);
                logger.Info(extraInfo, filter1, filter2, callerName);
            }
            catch (Exception ex)
            {
                stopwatch.Stop();
                extraInfo.Add("Elapsed", stopwatch.ElapsedMilliseconds.ToString());
                extraInfo.Add("ex", ex);
                logger.Error(ex, extraInfo, filter1, filter2, callerName);

                throw;
            }

            return responseText;
        }
        public async Task<T> GetAsync<T>(string url, int timeout = 6000, Encoding encoding = null, string filter1 = "", string filter2 = "", [CallerMemberName] string callerName = "")
        {
            var response = await GetAsync(url, timeout, encoding, filter1, filter2, callerName);
            return response.IsNullOrWhiteSpace() ? default(T) : response.FromJson<T>();
        }
        public async Task<string> PostAsync(string url, string postData, int timeout = 6000, string contentType = AppConstants.HttpMediaTypes.ApplicationJson, IDictionary<string, string> headers = null, Encoding encoding = null, string filter1 = "", string filter2 = "", string callerName = "")
        {
            Dictionary<string, dynamic> extraInfo = new Dictionary<string, dynamic>();
            extraInfo.Add("url", url);
            extraInfo.Add("method", "post");
            extraInfo.Add("requestText", postData);
            Stopwatch stopwatch = Stopwatch.StartNew();
            string responseText = string.Empty;
            try
            {
                var client = httpClientFactory.CreateClient();
                if (url.StartsWith(AppConstants.Protocols.HTTPS, StringComparison.CurrentCultureIgnoreCase))
                {
                    ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls;
                }
                if (timeout > 0)
                {
                    client.Timeout = TimeSpan.FromMilliseconds(timeout);
                }
                var httpContent = new StringContent(postData);
                httpContent.Headers.ContentType = new System.Net.Http.Headers.MediaTypeHeaderValue(contentType);
                if (headers?.Count > 0)
                {
                    foreach (var item in headers)
                    {
                        client.DefaultRequestHeaders.Add(item.Key, item.Value);
                    }
                }

                var response = await client.PostAsync(url, httpContent);
                var resultBytes = response.IsSuccessStatusCode ? await response.Content.ReadAsByteArrayAsync() : Array.Empty<byte>();

                responseText = (encoding ?? Encoding.UTF8).GetString(resultBytes);
                stopwatch.Stop();
                extraInfo.Add("Elapsed", stopwatch.ElapsedMilliseconds.ToString());
                extraInfo.Add("responseText", responseText);
                logger.Info(extraInfo, filter1, filter2, callerName);
            }
            catch (Exception ex)
            {
                stopwatch.Stop();
                extraInfo.Add("Elapsed", stopwatch.ElapsedMilliseconds.ToString());
                extraInfo.Add("ex", ex);
                logger.Error(ex, extraInfo, filter1, filter2, callerName);

                throw;
            }

            return responseText;
        }
        public async Task<T> PostAsync<T>(string url, string postData, int timeout = 6000, string contentType = AppConstants.HttpMediaTypes.ApplicationJson, IDictionary<string, string> headers = null, Encoding encoding = null, string filter1 = "", string filter2 = "", [CallerMemberName] string callerName = "")
        {
            var response = await PostAsync(url, postData, timeout, contentType, headers, encoding, filter1, filter2, callerName);
            return response.IsNullOrWhiteSpace() ? default(T) : response.FromJson<T>();
        }
        public async Task<TOut> PostAsync<TIn, TOut>(string url, TIn postData, int timeout = 6000, string contentType = AppConstants.HttpMediaTypes.ApplicationJson, IDictionary<string, string> headers = null, Encoding encoding = null, string filter1 = "", string filter2 = "", [CallerMemberName] string callerName = "")
            where TOut : class, new()
        {
            var response = await PostAsync(url, postData.ToJson(), timeout, contentType, headers, encoding, filter1, filter2, callerName);
            return response.IsNullOrWhiteSpace() ? default(TOut) : response.FromJson<TOut>();
        }
        public async Task<T> PostWithXmlResponseAsync<T>(string xmlString, string url, int timeout = 6000, string contentType = AppConstants.HttpMediaTypes.ApplicationXml, IDictionary<string, string> headers = null, Encoding encoding = null, string filter1 = "", string filter2 = "", [CallerMemberName] string callerName = "")
            where T : class, new()
        {
            var response = await PostAsync(url, xmlString, timeout, contentType, headers, encoding, filter1, filter2, callerName);
            return response.IsNullOrWhiteSpace() ? default(T) : response.FromXml<T>();
        }
        #endregion
    }
}
