﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.ComponentModel;
using System.Reflection;
using System.Text;

namespace Siger.WeComApi.Common.Helpers
{
    public class EnumHelper
    {
        private static ConcurrentDictionary<string, List<EnumItem>> dic = new ConcurrentDictionary<string, List<EnumItem>>();

        /// <summary>
        /// 获取一个枚举类型的列表，用于在dropdownList中显示
        /// </summary>
        /// <returns></returns>
        public static List<EnumItem> GetEnumItems(Type type)
        {
            if (!dic.ContainsKey(type.Name))
            {
                //System.Type type = typeof(T);
                System.Reflection.FieldInfo[] fields = type.GetFields();

                List<EnumItem> itemList = new List<EnumItem>(fields.Length);
                foreach (System.Reflection.FieldInfo fi in fields)
                {
                    if (fi.FieldType == type)
                    {
                        EnumItem item = new EnumItem();
                        item.Name = fi.Name;
                        item.Value = Convert.ToInt32(fi.GetRawConstantValue());
                        item.Description = EnumHelper.GetFieldDesc(fi);

                        if (string.IsNullOrEmpty(item.Description))
                        {
                            item.Description = item.Name;
                        }

                        itemList.Add(item);
                    }
                }
                dic[type.Name] = itemList;
            }
            return new List<EnumItem>(dic[type.Name]);
        }

        private static string GetFieldDesc(System.Reflection.FieldInfo field_info)
        {
            object[] attrs = field_info.GetCustomAttributes(typeof(System.ComponentModel.DescriptionAttribute), false);
            if ((attrs != null) && (attrs.Length > 0))
            {
                return (attrs[0] as System.ComponentModel.DescriptionAttribute).Description;
            }
            return field_info.Name;
        }

        /// <summary>
        /// 获取枚举值的描述
        /// </summary>
        /// <param name="enumType">指定的枚举类型</param>
        /// <param name="enumValue">枚举类型的值</param>
        /// <returns>枚举值的描述,需用DescriptionAttribute进行描述</returns>
        public static string GetDescription(Type enumType, int enumValue)
        {
            if (enumType == null)
            {
                throw new ArgumentNullException("enumType");
            }
            if (!enumType.IsEnum)
            {
                throw new ArgumentException("enumType不是枚举类型");
            }
            List<EnumItem> items = EnumHelper.GetEnumItems(enumType);
            EnumItem item = items.Find(p => p.Value == enumValue);
            if (item == null)
            {
                // throw new ArgumentException("enumType不包括此枚举值");
                return "未知";
            }
            return item == null ? enumValue.ToString() : item.Description;
        }

        /// <summary>
        /// 获取枚举值的描述
        /// </summary>
        /// <param name="enumType">指定的枚举类型</param>
        /// <param name="enumValue">枚举类型的值</param>
        /// <returns>枚举值的描述,需用DescriptionAttribute进行描述</returns>
        public static string GetDescription(Type enumType, int enumValue, string defDes)
        {
            if (enumType == null)
            {
                throw new ArgumentNullException("enumType");
            }
            if (!enumType.IsEnum)
            {
                throw new ArgumentException("enumType不是枚举类型");
            }
            List<EnumItem> items = EnumHelper.GetEnumItems(enumType);
            EnumItem item = items.Find(p => p.Value == enumValue);
            if (item == null)
            {
                // throw new ArgumentException("enumType不包括此枚举值");
                return defDes;
            }
            return item.Description;
        }

        /// <summary>
        /// 返回枚举项的描述信息。
        /// </summary>
        /// <param name="e">要获取描述信息的枚举项。</param>
        /// <returns>枚举项的描述信息,如果没有描述，则返回自身String。</returns>
        public static string GetEnumDesc(Enum e)
        {
            if (e == null)
            {
                return string.Empty;
            }
            Type enumType = e.GetType();
            DescriptionAttribute attr = null;

            // 获取枚举字段。
            FieldInfo fieldInfo = enumType.GetField(e.ToString());
            if (fieldInfo != null)
            {
                // 获取描述的属性。
                attr = Attribute.GetCustomAttribute(fieldInfo, typeof(DescriptionAttribute), false) as DescriptionAttribute;
            }

            // 返回结果
            return !string.IsNullOrEmpty(attr?.Description) ? attr.Description : e.ToString();
        }

        public static List<EnumberEntity> EnumToList<T>()
        {
            var list = new List<EnumberEntity>();

            foreach (var e in Enum.GetValues(typeof(T)))
            {
                EnumberEntity m = new EnumberEntity();
                object[] objArr = e.GetType().GetField(e.ToString()).GetCustomAttributes(typeof(DescriptionAttribute), true);
                if (objArr.Length > 0)
                {
                    DescriptionAttribute da = objArr[0] as DescriptionAttribute;
                    m.Description = da.Description;
                }
                m.EnumValue = Convert.ToInt32(e);
                m.EnumName = e.ToString();
                list.Add(m);
            }
            return list;
        }

        public static List<EnumberEntity> EnumToList(Type type)
        {
            List<EnumberEntity> list = new List<EnumberEntity>();
            foreach (var obj in Enum.GetValues(type))
            {
                list.Add(new EnumberEntity((Enum)obj));
            }
            return list;
        }
    }

  


    /// <summary>
    /// 枚举类项目
    /// </summary>
    public class EnumItem
    {
        /// <summary>
        /// 枚举名
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// 枚举值
        /// </summary>
        public int Value { get; set; }

        /// <summary>
        /// 枚举描述
        /// </summary>
        public string Description { get; set; }
    }

    public class EnumberEntity
    {
        /// <summary>  
        /// 枚举的描述  
        /// </summary>  
        public string Description { set; get; }

        /// <summary>  
        /// 枚举名称  
        /// </summary>  
        public string EnumName { set; get; }

        /// <summary>  
        /// 枚举对象的值  
        /// </summary>  
        public int EnumValue { set; get; }

        public EnumberEntity()
        {

        }

        public EnumberEntity(Enum um)
        {
            EnumValue = (int)Convert.ChangeType(um, typeof(int));
            EnumName = um.ToString();
            Description = EnumHelper.GetEnumDesc(um);
        }
    }
}
