﻿using System.Collections.Generic;
using Microsoft.AspNetCore.Mvc;
using Siger.Middlelayer.Common;
using Siger.Middlelayer.Repository;
using System.Linq;
using Siger.ApiCommon.Result;
using Siger.Middlelayer.Common.Helpers;
using Siger.Middlelayer.Repository.Entities;
using Siger.Middlelayer.Repository.Repositories.Interface;
using Siger.Middlelayer.Repository.Request;
using Siger.Middlelayer.Repository.Response;

namespace Siger.ApiConfig.Controller
{
    public class LevelController : BaseController
    {
        private readonly IUnitOfWork _unitOfWork;
        private readonly ISigerLevelRepository _levelRepository;
        private readonly ISigerProjectLevelRepository sigerProjectLevelRepository;

        public LevelController(IUnitOfWork unitOfWork, ISigerLevelRepository levelRepository, ISigerProjectLevelRepository sigerProjectLevelRepository)
        {
            _unitOfWork = unitOfWork;
            _levelRepository = levelRepository;
            this.sigerProjectLevelRepository = sigerProjectLevelRepository;
        }

        [HttpGet]
        public IActionResult GetAllLevel(int page, int pagesize)
        {
            var response = new List<ResponseGetLevel>();

            var levels = _levelRepository.GetPagedList(page, pagesize, q => q.status == (int) RowState.Valid);
            foreach (var level in levels.Data)
            {
                response.Add(Mapper<siger_level, ResponseGetLevel>.Map(level));
            }

            return new PagedObjectResult(response.OrderBy(q => q.sorting).ToList(), levels.Total, page, pagesize);
        }

        [HttpGet]
        public IActionResult GetLevel()
        {
            var response = new List<ResponseIdTitle>();
            var projectLevels = _levelRepository.GetList(q => q.status == (int)RowState.Valid);
            foreach (var projectLevel in projectLevels)
            {
                response.Add(Mapper<siger_level, ResponseIdTitle>.Map(projectLevel));
            }

            return new ObjectResult(response);
        }

        [HttpGet]
        public IActionResult GetLevels()
        {
            var projectLevels = sigerProjectLevelRepository.GetList(f => f.projectid == ProjectId && f.status == (int)RowState.Valid)
                                                            .Select(f=>new ResponseIdTitle
                                                            {
                                                                id=f.id,
                                                                title=f.title
                                                            }).OrderBy(f=>f.id);
            if(!projectLevels.Any())
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }
            return new ObjectResult(projectLevels);
        }

        [HttpGet]
        public IActionResult GetIdLevel(int id)
        {
            var level = _levelRepository.Get(id);
            if (level == null || level.status != (int)RowState.Valid)
            {
                throw new BadRequestException(RequestEnum.LevelNotFound);
            }

            var result = Mapper<siger_level, ResponseGetLevel>.Map(level);
            return new ObjectResult(result);
        }

        [HttpPost]
        public IActionResult AddLevel([FromBody]RequestAddLevel request)
        {
            var entity = _levelRepository.Get(q => q.title == request.title && q.status == (int) RowState.Valid);
            if (entity != null)
            {
                throw new BadRequestException(RequestEnum.TitleHasExist);
            }

            var level = new siger_level
            {
                title = request.title,
                description = request.description,
                parentid = 0,
                sorting = request.sorting
            };

            _levelRepository.Insert(level);

            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            throw new BadRequestException(CommonEnum.SystemExcetion);
        }

        [HttpPost]
        public IActionResult EditLevel([FromBody]RequestUpdateLevel request)
        {
            var level = _levelRepository.Get(request.id);
            if (level == null || level.status != (int)RowState.Valid)
            {
                throw new BadRequestException(RequestEnum.LevelNotFound);
            }
            var entity = _levelRepository.Get(q => q.title == request.title && q.id != level.id && q.status == (int)RowState.Valid);
            if (entity != null)
            {
                throw new BadRequestException(RequestEnum.TitleHasExist);
            }
            level.title = request.title;
            level.description = request.description;
            level.parentid = 0;
            level.sorting = request.sorting;

            _levelRepository.Update(level);

            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            throw new BadRequestException(CommonEnum.SystemExcetion);
        }

        [HttpPost]
        public IActionResult DeleteLevel(int id)
        {
            var level = _levelRepository.Get(id);
            if (level == null || level.status != (int)RowState.Valid)
            {
                throw new BadRequestException(RequestEnum.LevelNotFound);
            }

            level.status = (int) RowState.Invalid;
            _levelRepository.Update(level);

            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            throw new BadRequestException(CommonEnum.SystemExcetion);
        }
    }
}