﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Mvc;
using OfficeOpenXml.FormulaParsing.Excel.Functions.DateTime;
using Siger.Middlelayer.AccRepository.Entities;
using Siger.Middlelayer.AccRepository.Repositories.Interface;
using Siger.Middlelayer.AccRepository.Request;
using Siger.Middlelayer.AccRepository.Response;
using Siger.Middlelayer.Common;
using Siger.ApiCommon.Filters;
using Siger.Middlelayer.Common.Extensions;
using Siger.Middlelayer.Common.ModuleEnum;
using Siger.Middlelayer.Log;
using Siger.Middlelayer.Repository;
using Siger.Middlelayer.Repository.Repositories.Interface;
using Siger.Middlelayer.Repository.Response;
using Siger.Middlelayer.Share.Constant;
using Siger.Middlelayer.Share.Enum.ModuleEnum;

namespace Siger.ApiACC.Controllers
{
    [NoTokenValidateFilter]
    public class AutomationOperateController : BaseController
    {
        private readonly IUnitOfWork _unitOfWork;
        private readonly ISigerProjectLevelSectionRepository _sigerProjectLevelSection;
        private readonly ISigerDict _sigerDict;
        private readonly IAutomationMachineStatus _automationMachineStatus;
        private readonly ISigerProjectMachineAttributionRepository _sigerProjectMachineAttribution;
        private readonly IAutomationTaskListRepository _automationTaskList;
        private readonly IProductPlanDetails _planDetails;
        private readonly IProductPlanRepository _productPlan;
        private readonly IAutomationFixtureMonitor _automationFixtureMonitor;
        private readonly IAutomationFixtureToolsProductRepository _automationFixtureToolsProduct;
        private readonly IAutomationFixtureToolsRepository _automationFixtureTools;

        public AutomationOperateController(IUnitOfWork unitOfWork,ISigerProjectLevelSectionRepository sigerProjectLevelSection , ISigerDict sigerDict,IAutomationMachineStatus  automationMachineStatus,ISigerProjectMachineAttributionRepository sigerProjectMachineAttribution,IAutomationTaskListRepository automationTaskList,
            IProductPlanDetails planDetails,IProductPlanRepository productPlan,IAutomationFixtureMonitor automationFixtureMonitor ,IAutomationFixtureToolsProductRepository automationFixtureToolsProduct,IAutomationFixtureToolsRepository automationFixtureTools)
        {
            _unitOfWork = unitOfWork;
            _sigerProjectLevelSection = sigerProjectLevelSection;
            _sigerDict = sigerDict;
            _automationMachineStatus = automationMachineStatus;
            _sigerProjectMachineAttribution = sigerProjectMachineAttribution;
            _automationTaskList = automationTaskList;
            _planDetails = planDetails;
            _productPlan = productPlan;
            _automationFixtureMonitor = automationFixtureMonitor;
            _automationFixtureToolsProduct = automationFixtureToolsProduct;
            _automationFixtureTools = automationFixtureTools;
        }

        /// <summary>
        /// 获取字典配置 
        /// </summary>
        /// <param name="line">产线</param>
        /// <param name="type">类型：0：上料  1：检验</param>
        /// <returns></returns>
        [HttpGet]
        public IActionResult GetloadStation(int line,int type)
        {
            var stationDicts = _sigerDict.GetDataByCat(AccDictCost.Automation, ProjectId);
            if (!stationDicts.Any())
            {
                Logger.WriteLineInfo($"GetloadStation 未配置设备类型字典");
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }

            var loadStation=new List<string>();

            if (type==0)
            {
                var dictLoad = stationDicts.Where(s => s.dkey == DictKeyValConst.UploadloadStation);
                if (!dictLoad.Any())
                {
                    Logger.WriteLineInfo($"GetloadStation 未配置上下料工站字典");
                    throw new BadRequestException(CommonEnum.RecordNotFound);
                }
                loadStation = dictLoad.Select(s => s.dval).ToList();
            }else
            {
                var dictLoad = stationDicts.Where(s => s.dkey == DictKeyValConst.CleanStation);
                if (!dictLoad.Any())
                {
                    Logger.WriteLineInfo($"GetloadStation 未配置清洗工站字典");
                    throw new BadRequestException(CommonEnum.RecordNotFound);
                }
                loadStation = dictLoad.Select(s => s.dval).ToList();
            }
           

            var data = new List<ResponseLoadStation>();

            foreach (var station in loadStation)
            {
                var section = _sigerProjectLevelSection.Get(f => f.id == station.ToInt() && f.parentid==line && f.status == (int)RowState.Valid);
                if (section == null)
                    continue;
                var machintAttr = _sigerProjectMachineAttribution.Get(f => f.station == section.id);
                if(machintAttr==null)
                {
                    Logger.WriteLineInfo($"GetloadStation 未配置清洗工站字典");
                    continue;
                }
                var loadStatus = (int)Automation.MachineStatus.Waiting;
                var machineStatus = _automationMachineStatus.Get(f => f.section == section.id);
                if (machineStatus==null)
                {
                    _automationMachineStatus.Insert(new siger_automation_machine_status
                    {
                         projectId=ProjectId,
                         enable=1,
                         machineid=machintAttr.machine,
                         section=section.id,
                         updatetime=DateTime.Now
                    });
                }else
                {
                    loadStatus = machineStatus.status;
                }
                data.Add(new ResponseLoadStation { 
                 section=section.id,
                 title=section.title,
                 status=loadStatus
                });
            }
            return new ObjectResult(data);
        }


        /// <summary>
        /// 查询状态
        /// </summary>
        /// <param name="section"></param>
        /// <returns></returns>
        [HttpGet]
        public IActionResult GetLoadingState(int section)
        {

           
            var monitor = _automationFixtureMonitor.Get(f => f.section == section);
            var result = new ResponsePlanlFixtureInfo
            {
                OrderNumber = monitor.ordernumber,
                FixtureGuid=monitor.fixtureguid,
                FixtureName=monitor.fixturename,
                ProductCode=monitor.productCode,
                ProductName=monitor.productName,
                Sn=monitor.sn,
                status=monitor.status,
                Location=monitor.locationId,
            };
            return new ObjectResult(result);
        }

        /// <summary>
        /// 扫描
        /// </summary>
        /// <param name="code"></param>
        /// <returns></returns>
        [HttpGet]
        public IActionResult ScanCode(string code)
        {
            var result = _automationFixtureToolsProduct.GetPlanFixtureInfo(ProjectId, code);
            if (result==null)
            {
                throw new BadRequestException(AccEnum.FixtureOrderNulll);
            }
            var sn = _automationTaskList.CreateRandonSn(result.ProductCode);
            result.Sn = sn;
            return new ObjectResult(result);
        }
        /// <summary>
        /// 准备上料 -生成指令 load
        /// </summary>
        /// <param name="loading"></param>
        /// <returns></returns>
        [HttpPost]
        public IActionResult Loading([FromBody]requestAutomationLoading loading)
        {

            var machineAttr = _sigerProjectMachineAttribution.Get(f => f.station == loading.section && f.status == (int)RowState.Valid);
            if (machineAttr==null)
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }
            var machineStatus = _automationMachineStatus.Get(f => f.machineid == machineAttr.machine);
            if (machineStatus == null)
            {
                // throw new BadRequestException(CommonEnum.RecordNotFound);
                //上料设备状态与PLC 无关，
                //需要手动插入
                _automationMachineStatus.Insert(new siger_automation_machine_status
                {
                    enable = 1,
                    section = loading.section,
                    machineid = machineAttr.machine,
                    projectId = ProjectId,
                    status = (int)Automation.MachineStatus.Produce,
                    updatetime = DateTime.Now
                });
            }
            else
            {
                if (machineStatus.enable == 0)
                {
                    throw new BadRequestException(AccEnum.MachineDisable);
                }
                if (machineStatus.status != (int)Automation.MachineStatus.Waiting)
                {
                    throw new BadRequestException(AccEnum.MachineBusy);
                }
            }
            if (!_automationTaskList.CanTask(ProjectId,loading.section))
            {
                throw new BadRequestException(AccEnum.TaskProcessing);
            }

            var plandts = _planDetails.Get(f => f.projectId == ProjectId && f.OrderNumber == loading.code);
            if (plandts==null)
            {
                throw new BadRequestException(CncEnum.PlanHasExist);
            }
            var plan = _productPlan.Get(f => f.id == plandts.PlanId);
            if (plan == null)
            {
                throw new BadRequestException(CncEnum.PlanHasExist);
            }
            //检查工装
            var fixtureToolsObj = _automationFixtureTools.GetProductFixtureLocation(ProjectId,loading.guid);
            //var fixtureToolsObj = _automationFixtureToolsProduct.GetFixtureInfoByProductCode(ProjectId, plan.product_code);
            if (fixtureToolsObj == null)
            {
                throw new BadRequestException(AccEnum.FixtureToolsIsNone);
            }
            var taskNo = _automationTaskList.CrateTaskNumber(Automation.TaskTrigerType.Manual);
            var sn = _automationTaskList.CreateRandonSn(plan.product_code);

            _automationTaskList.Insert(new siger_automation_task_list
            {
                 no=taskNo,
                 action= Automation.TaskAction.Step_LK_SXLW,
                 actiontype= Automation.ExcueType.None,
                 triggertime=DateTime.MinValue,
                 tasktype= Automation.TaskActionType.Load,
                 operater=UserId,
                 operatetime=DateTime.Now,
                 sectionid=loading.section,
                 send=0,
                 status=1,
                 completetime=DateTime.MinValue,
                 trigger= Automation.TaskTrigerType.Manual,
                 projectId=ProjectId,
                 productid=plan.product_id,
                 sn=sn,
                 ordercode=plandts.OrderNumber,
                 fixtureguid= fixtureToolsObj.FixtureGuid,
                 locationid=fixtureToolsObj.Location,
                 productcode=plan.product_code,
                 processid=0,
                 programnumber="",
                 remark ="手动任务-准备上料",
                 
            });


            var monitor = _automationFixtureMonitor.Get(f => f.fixtureguid == fixtureToolsObj.FixtureGuid);
            if (monitor==null)
            {
                _automationFixtureMonitor.Insert(new siger_automation_fixture_tools_monitor
                {
                     fixtureguid=fixtureToolsObj.FixtureGuid,
                     fixturename=fixtureToolsObj.FixtureName,
                     projectId=ProjectId,
                     section=loading.section,
                     sn=sn,
                     createtime=DateTime.Now,
                     updatetime=DateTime.Now,
                     status=(int)Automation.MachineStatus.Produce,
                     productCode=plan.product_code,
                     productName=plan.product_name,
                     ordernumber=plandts.OrderNumber,
                     locationId= fixtureToolsObj.Location
                });
            }
            else
            {
                monitor.sn = sn;
                monitor.section = loading.section;
                monitor.updatetime = DateTime.Now;
                monitor.status = (int)Automation.MachineStatus.Produce;
                monitor.productName = plan.product_name;
                monitor.productCode = plan.product_code;
                monitor.ordernumber = plandts.OrderNumber;
                monitor.locationId = fixtureToolsObj.Location;
               
                _automationFixtureMonitor.Update(monitor);
            }


            if (_unitOfWork.Commit() > 0)
            {
                Logger.WriteLineError($"手动任务创建成功-{Siger.Middlelayer.Common.Helpers.EnumHelper.GetEnumDesc(Automation.TaskAction.Step_LK_SXLW)}");
                return new ObjectResult(CommonEnum.Succefull);
            }
            else
            {
                throw new BadRequestException(CommonEnum.Fail);
            }
        }

        /// <summary>
        /// 安装完成 -生成指令 unload
        /// </summary>
        /// <param name="assemble"></param>
        /// <returns></returns>
        [HttpPost]
        public IActionResult CompalateAssemble([FromBody]requestAutoCompalateAssemble assemble)
        {
            var machineStatus = _automationMachineStatus.Get(f => f.section == assemble.section);
            if (machineStatus==null)
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }
            if (machineStatus.enable == 0)
            {
                throw new BadRequestException(AccEnum.MachineDisable);
            }
            if (machineStatus.status == (int)Automation.MachineStatus.Waiting)
            {
                throw new BadRequestException(AccEnum.MachineOnFree);
            }
            if (machineStatus.status == (int)Automation.MachineStatus.Complated)
            {
                throw new BadRequestException(AccEnum.MachineProCompalate);
            }

            if (!_automationTaskList.CanTask(ProjectId, assemble.section))
            {
                throw new BadRequestException(AccEnum.TaskProcessing);
            }
            //创建 安装完成动作
            var monitor = _automationFixtureMonitor.Get(f => f.projectId == ProjectId && f.section == assemble.section);
            if (monitor==null)
            {
                throw new BadRequestException(AccEnum.MonitorNotfound);
            }

            var taskNo = _automationTaskList.CrateTaskNumber(Automation.TaskTrigerType.Manual);
            _automationTaskList.Insert(new siger_automation_task_list
            {
                no = taskNo,
                action = Automation.TaskAction.Step_SXLW_LK,
                actiontype = Automation.ExcueType.None,
                triggertime = DateTime.MinValue,
                tasktype = Automation.TaskActionType.Load,
                operater = UserId,
                operatetime = DateTime.Now,
                sectionid = assemble.section,
                send = 0,
                status = 1,
                completetime = DateTime.MinValue,
                trigger = Automation.TaskTrigerType.Manual,
                projectId = ProjectId,
                productid = 0,
                sn = monitor.sn,
                ordercode = monitor.ordernumber,
                fixtureguid = monitor.fixtureguid,
                locationid = monitor.locationId,
                productcode = monitor.productCode,
                processid = 0,
                programnumber = "",
                remark = "手动任务-安装完成",

            });


            if (_unitOfWork.Commit() > 0)
            {
                Logger.WriteLineError($"手动任务创建成功-{Siger.Middlelayer.Common.Helpers.EnumHelper.GetEnumDesc(Automation.TaskAction.Step_SXLW_LK)}");
                return new ObjectResult(CommonEnum.Succefull);
            }
            else
            {
                throw new BadRequestException(CommonEnum.Fail);
            }
        }

        /// <summary>
        /// 准备下料
        /// </summary>
        /// <param name="unloading"></param>
        /// <returns></returns>
        [HttpPost]
        public IActionResult Unloading([FromBody]requestAutomationUnloading unloading)
        {

            return new ObjectResult(1);
        }

        /// <summary>
        /// 拆卸完成
        /// </summary>
        /// <param name="disassemble"></param>
        /// <returns></returns>
        [HttpPost]
        public IActionResult Disassemble([FromBody]Requestdisassemble disassemble)
        {

            return new ObjectResult(1);
        }
    }
}
