﻿using System;
using Microsoft.AspNetCore.Mvc;
using Siger.ApiCommon.Exceptions;
using Siger.ApiCommon.Result;
using Siger.Middlelayer.Common;
using Siger.Middlelayer.Common.Extensions;
using Siger.Middlelayer.Repository;
using Siger.Middlelayer.Repository.Entities;
using Siger.Middlelayer.Repository.Repositories.Interface;
using Siger.Middlelayer.Repository.Request;

namespace Siger.ApiConfig.Controller
{
    public class LineStorageAuthorityController : BaseController
    {
        private readonly IUnitOfWork _unitOfWork;
        private readonly ISigerProjectLineStorageAuthorityRepository _authRepository;
        public LineStorageAuthorityController(IUnitOfWork unitOfWork, ISigerProjectLineStorageAuthorityRepository authRepository)
        {
            _unitOfWork = unitOfWork;
            _authRepository = authRepository;
        }

        [HttpGet]
        public IActionResult GetStorageAuthorities(int storage_id, int  page = 1, int pagesize = 10)
        {
            var response = _authRepository.GetPagedAuth(storage_id, ProjectId, page, pagesize);

            return new PagedObjectResult(response.Data, response.Total, page, pagesize);
        }

        [HttpPost]
        public IActionResult AddStorageAuthority([FromBody]RequestAddStorageAuthority request)
        {
            //一个人只能管理一个线边柜
            var isExist = _authRepository.IsExist(q => q.user_id == request.user_id.ToInt() &&
                q.project_id == ProjectId && q.status == (int) RowState.Valid);
            if (isExist)
            {
                throw new BadRequestException(CommonEnum.RecordExits);
            }
            var entity = new siger_project_line_storage_authority
            {
                create_time = DateTime.Now,
                creator_id = UserId,
                manage_authority = request.manager_auth.ToInt(),
                use_authority = request.use_auth.ToInt(),
                user_id = request.user_id.ToInt(),
                section_id = request.section_id.ToInt(),
                project_id = ProjectId,
                storage_id = request.storage_id.ToInt()
            };

            _authRepository.Insert(entity);
            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            throw new BadRequestException(CommonEnum.Fail);
        }

        [HttpGet]
        public IActionResult GetStorageAuthorityItem(int id)
        {
            var data = _authRepository.GetStorageAuthority(id, ProjectId);
            if (data == null)
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }

            return new ObjectResult(data);
        }

        [HttpPost]
        public IActionResult EditStorageAuthority([FromBody]RequestEditStorageAuthority request)
        {
            var isExist = _authRepository.IsExist(q =>
                q.user_id == request.user_id.ToInt() &&
                q.project_id == ProjectId && q.status == (int)RowState.Valid && q.id != request.id);
            if (isExist)
            {
                throw new BadRequestException(CommonEnum.RecordExits);
            }

            var entity = _authRepository.Get(request.id);
            if (entity == null || entity.status == (int)RowState.Invalid)
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }

            entity.manage_authority = request.manager_auth.ToInt();
            entity.use_authority = request.use_auth.ToInt();
            _authRepository.Update(entity);
            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            throw new BadRequestException(CommonEnum.Fail);
        }

        [HttpPost]
        public IActionResult DeleteStorageAuthority(int id)
        {
            var entity = _authRepository.Get(id);
            if (entity == null || entity.status == (int) RowState.Invalid)
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }

            entity.status = (int) RowState.Invalid;
            _authRepository.Update(entity);
            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            throw new BadRequestException(CommonEnum.Fail);
        }
    }
}