﻿using Microsoft.AspNetCore.Mvc;
using NPOI.SS.Formula.Functions;
using System.IO;
using System;
using System.Linq;
using System.Text;
using Siger.ApiCommon;
using Siger.ApiCommon.Result;
using Siger.Middlelayer.Common;
using Siger.Middlelayer.Common.Helpers;
using Siger.Middlelayer.Common.ModuleEnum;
using Siger.Middlelayer.EsopRepository.Entities;
using Siger.Middlelayer.EsopRepository.Repositories.Interface;
using Siger.Middlelayer.EsopRepository.Request;
using Siger.Middlelayer.EsopRepository.Response;
using Siger.Middlelayer.Repository.Repositories.Interface;
using System.Collections.Generic;
using Siger.ApiCommon.Filters;
using Siger.Middlelayer.Common.AppSettings;
using Siger.ApiCommon.Utilities;

namespace Siger.ApiEsop.Controllers
{
    public class EsopDocumentSearchController : BaseController
    {
        private readonly IEsopDocumentBindingRepository _documentBindingRepository;
        private readonly ISigerProjectMachineRepository _machineRepository;
        private readonly IEsopDocumentSetRepository _esopDocumentSetRepository;
        private readonly IEsopDocumentCreateRepository _documentCreateRepository;
        private readonly IEsopDocumentHistoryRepository _esopDocumentHistoryRepository;

        public EsopDocumentSearchController(IEsopDocumentBindingRepository documentSetRepository, ISigerProjectMachineRepository machineRepository, IEsopDocumentSetRepository esopDocumentSetRepository
            , IEsopDocumentCreateRepository documentCreateRepository, IEsopDocumentHistoryRepository esopDocumentHistoryRepository)
        {
            _documentBindingRepository = documentSetRepository;
            _machineRepository = machineRepository;
            _esopDocumentSetRepository = esopDocumentSetRepository;
            _documentCreateRepository = documentCreateRepository;
            _esopDocumentHistoryRepository = esopDocumentHistoryRepository;
        }

        /// <summary>
        /// 文档使用记录, 按工艺路径查询
        /// </summary>
        /// <param name="starttime"></param>
        /// <param name="endtime"></param>
        /// <param name="product_id"></param>
        /// <param name="level_id"></param>
        /// <param name="page"></param>
        /// <param name="pagesize"></param>
        /// <returns></returns>
        [HttpGet]
        public IActionResult GetPagedDocumentsUseRecord(string starttime, string endtime, int product_id, int level_id,
            int page = 1, int pagesize = 10)
        {
            var data = _documentBindingRepository.SearchDocumentBinding(starttime, endtime, product_id, level_id, page,
                pagesize, ProjectId);

            return new PagedObjectResult(data.Data, data.Total, page, pagesize);
        }

        /// <summary>
        /// 文档使用记录, 按设备查询
        /// </summary>
        /// <param name="starttime"></param>
        /// <param name="endtime"></param>
        /// <param name="section_id"></param>
        /// <param name="keyword">文档名称和类型</param>
        /// <param name="page"></param>
        /// <param name="pagesize"></param>
        /// <returns></returns>
        [HttpGet]
        public IActionResult SearchMachinePagedBindings(string starttime, string endtime, int section_id, string keyword,
            int page = 1, int pagesize = 10)
        {
            var machineIds = _machineRepository.GetNCLevelSectionMachineIds(section_id, ProjectId);

            var data = _documentBindingRepository.SearchDocumentBindingByMachine(starttime, endtime, machineIds, keyword, page,
                pagesize, ProjectId);

            return new PagedObjectResult(data.Data, data.Total, page, pagesize);
        }

        /// <summary>
        /// 文档现场展示
        /// </summary>
        /// <param name="section_id"></param>
        /// <param name="type_id"></param>
        /// <param name="document_name"></param>
        /// <returns></returns>
        [HttpGet]
        public IActionResult SearchDocuments(int section_id, int type_id, string document_name)
        {
            var machineIds = _machineRepository.GetNCLevelSectionMachineIds(section_id, ProjectId);

            var data = _documentBindingRepository.ShowDocumentBindings(machineIds, type_id, document_name, ProjectId);

            return new ObjectResult(data);
        }
        /// <summary>
        /// 标准规程 内部手册 文档查询
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        public IActionResult GetSensorDocs([FromBody]RequestSensorDocs request)
        {
            var ret = new List<string>();
            var typeList = _esopDocumentSetRepository.GetList(f => f.project_id == ProjectId && f.status == 1 && f.name.Equals(request.type)).Select(s => s.id);
            if (typeList.Any())
            {
                var file = _documentCreateRepository.GetList(f => f.project_id == ProjectId && f.status == 1 && typeList.Contains(f.type_id)).Select(s => s.id);
                ret = _esopDocumentHistoryRepository.GetList(f => f.status != 0 && file.Contains(f.document_id) && f.is_valid == 1).Select(s => s.version_name).Distinct().ToList();
            }
            return new ObjectResult(ret);
        }

        /// <summary>
        /// 获取ESOP 文档 图纸
        /// </summary>
        /// <param name="station">工位ID</param>
        /// <param name="productCode">产品Code</param>
        /// <returns></returns>
        [HttpGet]
        public IActionResult GetFileUrlInfo(int station, string productCode)
        {
            if (station==0)
            {
                throw new BadRequestException(AccEnum.Station_Null);
            }

            if (string.IsNullOrEmpty(productCode))
            {
                throw new BadRequestException(AccEnum.ProductInfo_Record_Null);
            }
            var response = GetEsopFile(ProjectId, station, productCode);

            return new ObjectResult(response);
        }
        /// <summary>
        /// C/S 端 获取ESOP 文档 图纸
        /// </summary>
        /// <param name="projectId"></param>
        /// <param name="station"></param>
        /// <param name="productCode"></param>
        /// <returns></returns>
        [HttpGet]
        [NoTokenValidateFilter]
        public IActionResult GetClientFileUrlInfo(int projectId,int station,string productCode)
        {
            var response = GetEsopFile(projectId, station, productCode);
            if (!response.Any())
                throw new ClientException(EnumHelper.GetEnumDesc(CommonEnum.RecordNotFound));

            return new ObjectResult(response);
        }

        /// <summary>
        /// ESOP 文档 图纸
        /// </summary>
        /// <param name="projectId"></param>
        /// <param name="station"></param>
        /// <param name="productCode"></param>
        /// <returns></returns>
        IEnumerable<ResponseFileInfo> GetEsopFile(int projectId,int station,string productCode)
        {
            var fileUrl = _documentCreateRepository.GetFileUrl(station, productCode, projectId);
           
            return fileUrl;
        }
    }
}