﻿using System.Linq;
using Microsoft.AspNetCore.Mvc;
using Siger.ApiCommon.Exceptions;
using Siger.ApiTPM.Result;
using Siger.Middlelayer.Common;
using Siger.Middlelayer.Common.Extensions;
using Siger.Middlelayer.Common.Helpers;
using Siger.Middlelayer.Repository;
using Siger.Middlelayer.Repository.Repositories.Interface;
using Siger.Middlelayer.TpmRepository.Entities;
using Siger.Middlelayer.TpmRepository.Repositories.Interface;
using Siger.Middlelayer.TpmRepository.Request;

namespace Siger.ApiTPM.Controllers
{
    public class RepairReportController : BaseController
    {
        private readonly IUnitOfWork _unitOfWork;
        private readonly IRepairReportRepository _reportRepository;
        private readonly ISigerSystemConfigRepository _configRepository;

        public RepairReportController(IUnitOfWork unitOfWork, IRepairReportRepository reportRepository, ISigerSystemConfigRepository configRepository)
        {
            _unitOfWork = unitOfWork;
            _reportRepository = reportRepository;
            _configRepository = configRepository;
        }

        [HttpGet]
        public IActionResult GetSetTime(int machinelevel = 1)
        {
            var sets = _reportRepository.GetList(q =>
                q.machinelevel == machinelevel && q.projectid == ProjectId && q.status == (int) RowState.Valid)
                .Select(m => new
                {
                    m.repeat_time,
                    m.end_level,
                    m.space_time,
                    m.start_level,
                    m.type
                }).ToList();

            var config = _configRepository.Get(q => q.key == SystemConfigKey.MessageRepeatTime && q.status == (int)RowState.Valid);

            var result = new RepairReportResult
            {
                data = sets,
                //time = config == null ? "15" : config.value
                time = sets.Any() ? sets.FirstOrDefault().repeat_time : 0
            };

            return result;
        }

        [HttpGet]
        public IActionResult GetSetLevel(int machinelevel = 1)
        {
            var levels = _reportRepository.GetSetLevels(machinelevel, ProjectId);
            var result = new SetLevelResult();
            result.data.Add("level1", levels.Where(m => m.level == 1).ToList());
            result.data.Add("level2", levels.Where(m => m.level == 2).ToList());
            result.data.Add("level3", levels.Where(m => m.level == 3).ToList());
            result.data.Add("level4", levels.Where(m => m.level == 4).ToList());

            return result;
        }

        [HttpPost]
        public IActionResult SetTime([FromBody]RequestSetTime request)
        {
            if (string.IsNullOrWhiteSpace(request.machinelevel))
            {
                throw new ServerException(100184);
            }
            if (request.data == null)
            {
                throw new BadRequestException(RequestEnum.ParameterError);
            }

            foreach (var setTime in request.data)
            {
                if (setTime.space_time <= 2)
                {
                    throw new ServerException(900794);
                }
            }

            foreach (var setTime in request.data)
            {
                var exist = _reportRepository.Get(q =>
                    q.projectid == ProjectId && q.status == (int)RowState.Valid && q.machinelevel == request.machinelevel.ToInt()
                    && q.type == setTime.type && q.start_level == setTime.start_level && q.end_level == setTime.end_level);
                if (exist != null)
                {
                    exist.start_level = setTime.start_level;
                    exist.end_level = setTime.end_level;
                    exist.space_time = setTime.space_time;
                    exist.create_time = UnixTimeHelper.GetNow();
                    exist.create_mid = UserId;
                    exist.repeat_time = request.message_repeat_time.ToInt();
                    _reportRepository.Update(exist);
                }
                else
                {
                    var entity = new siger_project_repair_report
                    {
                        projectid = ProjectId,
                        create_time = UnixTimeHelper.GetNow(),
                        create_mid = UserId,
                        end_level = setTime.end_level,
                        start_level = setTime.start_level,
                        machinelevel = request.machinelevel.ToInt(),
                        space_time = setTime.space_time,
                        type = setTime.type,
                        repeat_time= request.message_repeat_time.ToInt()

                    };
                    _reportRepository.Insert(entity);
                }
            }

           // var setResult = _reportRepository.SetMessageRepeatTime(request.message_repeat_time);
            if ( _unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            throw new BadRequestException(CommonEnum.Fail);
        }

        [HttpPost]
        public IActionResult SetLevel([FromBody]RequestSetLevel request)
        {
            if (request.level == 0)
            {
                throw new ServerException(100205);
            }

            if (string.IsNullOrWhiteSpace(request.machinelevel))
            {
                throw new ServerException(100184);
            }

            var addResult = _reportRepository.AddLevels(request, ProjectId);
            
            return new ObjectResult(addResult > 0 ? CommonEnum.Succefull : CommonEnum.Fail);
        }
    }
}