﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.EntityFrameworkCore;
using Siger.Middlelayer.CncRepository.Response;
using Siger.Middlelayer.Common;
using Siger.Middlelayer.Common.Extensions;
using Siger.Middlelayer.Common.ModuleEnum;
using Siger.Middlelayer.Dapper.ResultData;
using Siger.Middlelayer.Repository.Data;
using Siger.Middlelayer.Repository.Entities;
using Siger.Middlelayer.Repository.Response;
using Siger.Middlelayer.Share.Models;

namespace Siger.Middlelayer.Repository.Repositories
{
    internal abstract class ApiConfigRepositoryBase<TEntity> : RepositoryBase<TEntity> where TEntity : ConfigEntityBase
    {
        private readonly ApiConfigDbContext _context;
        protected ApiConfigRepositoryBase(ApiConfigDbContext context) : base(context)
        {
            _context = context;
        }

        public int GetProjectLanguage(int projectId)
        {
            var project = _context.siger_project.FirstOrDefault(t => t.id == projectId);
            return project?.language ?? 0;
        }

        public IEnumerable<int> GetLevelSectionIds(int id, int projectid)
        {
            var list = new List<int>();
            var query = GetSonLevelSections(id, projectid);

            list.Add(id);
            foreach (var section in query.ToList())
            {
                list.Add(section.id);
            }

            return list;
        }


        public IEnumerable<string> GetLevelSectionTitles(int id, int projectid)
        {
            var list = new List<string>();

            var query = GetParentLevelSections(id, projectid);

            foreach (var section in query.ToList().OrderBy(q => q.levelid))
            {
                list.Add(section.title);
            }

            return list;
        }

        public IEnumerable<siger_project_level_section> GetParentLevelSections(int id, int projectid)
        {
            var query = from c in _context.siger_project_level_section
                        where c.id == id && c.projectid == projectid && c.status == (int)RowState.Valid
                        select c;

            return query.ToList().Concat(query.ToList().SelectMany(t => GetParentLevelSections(t.parentid, projectid)));
        }

        public IEnumerable<int> GetLevelSectionMachineIds(int id, int projectid)
        {
            var list = new List<int>();
            var query = GetSonLevelSections(id, projectid);

            list.Add(id);
            foreach (var section in query.ToList())
            {
                list.Add(section.id);
            }

            var querylist = from q in _context.siger_project_machine_attribution
                            join m in _context.siger_project_machine on q.machine equals m.id into ma
                            from m in ma.DefaultIfEmpty()
                            where list.Contains(q.station) && q.status == (int)RowState.Valid &&
                                  m.status == (int)RowState.Valid
                                  && m.projectid == projectid
                            orderby m.sorting
                            select q.machine;

            return querylist.ToList();
        }

        public IEnumerable<LevelSectionTree> GetLevelSectionMachines(int projectid)
        {
            var list = new List<LevelSectionTree>();
            var query = GetSonLevelSections(0, projectid);

            foreach (var section in query.ToList())
            {
                list.Add(new LevelSectionTree { id = section.id, name = section.title, pid = section.parentid, open = true });
            }

            var querylist = from q in _context.siger_project_machine_attribution
                            join m in _context.siger_project_machine on q.machine equals m.id into ma
                            from m in ma.DefaultIfEmpty()
                            where list.Select(s => s.id).Contains(q.station) && q.status == (int)RowState.Valid &&
                                  m.status == (int)RowState.Valid
                                  && m.projectid == projectid
                            orderby m.sorting
                            select new LevelSectionTree
                            {
                                id = m.id,
                                name = m.title,
                                pid = q.station,
                            };
            foreach (var levelSectionTree in querylist.ToList())
            {
                list.Add(new LevelSectionTree { id = levelSectionTree.id, name = levelSectionTree.name, pid = levelSectionTree.pid, open = true });
            }

            return list;
        }

        public IEnumerable<siger_project_level_section> GetSonLevelSections(int parentId, int projectid)
        {
            if (parentId == 0)
            {
                return GetAllSections(projectid);
            }
            var query = (from c in _context.siger_project_level_section
                where c.id == parentId && c.projectid == projectid && c.status == (int)RowState.Valid
                select c).FirstOrDefault();
            if (query != null && query.parentid == 0)
            {
                //return GetAllSections(projectid);
                return GetSonLevelSectionsLop(query.id, projectid);
            }
            return GetSonLevelSections(parentId, GetAllSections(projectid));
        }

        private IEnumerable<siger_project_level_section> GetSonLevelSectionsLop(int parentId, int projectid)
        {
            var query = from c in _context.siger_project_level_section
                where c.parentid == parentId && c.projectid == projectid && c.status == (int)RowState.Valid
                select c;

            return query.ToList().Concat(query.ToList().SelectMany(t => GetSonLevelSectionsLop(t.id, projectid)));
        }

        private IEnumerable<siger_project_level_section> GetSonLevelSections(int parentId, List<siger_project_level_section> sectionList)
        {
            var query = sectionList.Where(f => f.parentid == parentId);
            return query.ToList().Concat(query.ToList().SelectMany(t => GetSonLevelSections(t.id, sectionList)));
        }

        private List<siger_project_level_section> GetAllSections(int projectid)
        {
            return _context.siger_project_level_section.Where(q => q.projectid == projectid && q.status == (int)RowState.Valid).AsNoTracking().ToList();
        }

        public IEnumerable<siger_project_level_section> GetLevelSections(int id, int projectid)
        {
            var list = new List<siger_project_level_section>();
            var query = GetSonLevelSections(id, projectid);

            foreach (var section in query.ToList())
            {
                list.Add(section);
            }

            var self = _context.siger_project_level_section.FirstOrDefault(q => q.id == id);
            if (self != null && !list.Select(m => m.id).Contains(self.id))
            {
                list.Add(self);
            }

            return list;
        }

        public IEnumerable<ResponseKeyValue> GetDictKeyValues(int projectId, string key)
        {
            var result = new List<ResponseKeyValue>();
            var query = _context.siger_tr_dict.Where(q =>
                q.projectId == projectId && q.cat == key && q.status == (int)RowState.Valid);
            if (query.Any())
            {
                foreach (var dict in query.ToList())
                {
                    result.Add(new ResponseKeyValue(dict.dkey, dict.dval));
                }
            }

            return result;
        }

        public IEnumerable<MachineData> GetLevelSectionMachines(int id, int projectid)
        {
            var list = new List<int>();
            var query = GetSonLevelSections(id, projectid);

            list.Add(id);
            foreach (var section in query.ToList())
            {
                list.Add(section.id);
            }

            var querylist = from q in _context.siger_project_machine_attribution
                            join m in _context.siger_project_machine on q.machine equals m.id
                            join se in _context.siger_project_level_section on q.station equals se.id
                            where list.Contains(q.station) && q.status == (int)RowState.Valid &&
                                  m.status == (int)RowState.Valid && se.status == (int)RowState.Valid
                                  && m.projectid == projectid
                                  orderby m.sorting,m.id
                            select new MachineData
                            {
                                //用于判断只要nc得
                                category = m.category,
                                section_id = q.station,
                                machine_id = m.id,
                                machine_name = m.title,
                                machine_code = m.code,
                                lastSectionTitle = se.title,
                                sorting = m.sorting
                            };

            return querylist.OrderBy(m => m.sorting).ToList();
        }

        public IEnumerable<MachineData> GetLevelSectionNames(int id, int projectid)
        {
            var list = new List<int>();
            var query = GetSonLevelSections(id, projectid);

            list.Add(id);
            foreach (var section in query.ToList())
            {
                list.Add(section.id);
            }

            var querylist = from q in _context.siger_project_machine_attribution
                join m in _context.siger_project_machine on q.machine equals m.id into ma
                from m in ma.DefaultIfEmpty()
                join se in _context.siger_project_level_section on q.station equals se.id
                join se2 in _context.siger_project_level_section on se.parentid equals se2.id
                where list.Contains(q.station) && q.status == (int)RowState.Valid  && m.attribution < 4
                orderby m.sorting
                select new MachineData
                {
                    machine_id = m.id,
                    machine_name = m.title,
                    machine_code = m.code,
                    section_id = se.id,
                    lastSectionTitle = se.title,
                    lastSecondSectionTitle = se2.title,
                    sorting = m.sorting
                };

            return querylist.OrderBy(m => m.sorting).ToList();
        }

        public IEnumerable<MachineData> GetLevelSectionNames(IEnumerable<int> machineIds, int projectid)
        {
            var querylist = from q in _context.siger_project_machine_attribution
                            join m in _context.siger_project_machine on q.machine equals m.id
                            join se in _context.siger_project_level_section on q.station equals se.id
                            join se2 in _context.siger_project_level_section on se.parentid equals se2.id
                            where machineIds.Contains(q.station) && q.status == (int)RowState.Valid &&
                                  m.status == (int)RowState.Valid && se.status == (int)RowState.Valid
                                  && m.projectid == projectid
                            select new MachineData
                            {
                                machine_id = m.id,
                                machine_name = m.title,
                                machine_code = m.code,
                                section_id = se.id,
                                lastSectionTitle = se.title,
                                lastSecondSectionTitle = se2.title,
                                sorting = m.sorting
                            };

            return querylist.OrderBy(m => m.sorting).ToList();
        }

        public siger_project_level_section GetLevelSection(int id, int projectid)
        {
            return _context.siger_project_level_section.FirstOrDefault(t => t.id == id && t.projectid == projectid && t.status == (int)RowState.Valid);
        }

        public IEnumerable<siger_project_level_section> GetSecondLevelSection(int projectid)
        {
            var query = from q in _context.siger_project_level_section
                        join qq in _context.siger_project_level_section on q.id equals qq.parentid
                        where q.parentid == 0 && q.projectid == projectid
                        select qq;
            return query.AsEnumerable();
        }

        public siger_project_machine_attribution GetMachineAttributionByMachineId(int machineId)
        {
            return _context.siger_project_machine_attribution.FirstOrDefault(t => t.machine == machineId && t.status == (int)RowState.Valid);
        }

        public IEnumerable<siger_project_level_section> GetParentSelfLevelSections(int parentid, int projectid)
        {
            var query = from c in _context.siger_project_level_section
                        where c.id == parentid && c.projectid == projectid && c.status == (int)RowState.Valid
                        select c;

            return query.ToList().Concat(query.ToList().SelectMany(t => GetParentLevelSections(t.parentid, projectid)));
        }

        public IEnumerable<int> GetMachineIdsBySectionIds(IEnumerable<int> sectionIds, int projectid)
        {
            return from q in _context.siger_project_machine_attribution
                   join m in _context.siger_project_machine on q.machine equals m.id
                   where m.projectid == projectid && q.status == (int)RowState.Valid &&
                         m.status == (int)RowState.Valid && sectionIds.Contains(q.station)
                   select q.machine;
        }

        public IEnumerable<siger_project_level_section> GetLevelSectionsBySectionIds(IEnumerable<int> sectionIds, int projectid)
        {
            return _context.siger_project_level_section.Where(t => sectionIds.Contains(t.id) && t.projectid == projectid && t.status == (int)RowState.Valid);
        }


        public IEnumerable<int> GetNCLevelSectionMachineIds(int id, int projectid)
        {
            var list = new List<int>();
            var query = GetSonLevelSections(id, projectid);

            list.Add(id);
            foreach (var section in query.ToList())
            {
                list.Add(section.id);
            }

            var querylist = from q in _context.siger_project_machine_attribution
                            join m in _context.siger_project_machine on q.machine equals m.id 
                            where list.Contains(q.station) && q.status == (int)RowState.Valid &&
                                  m.status == (int)RowState.Valid && m.projectid == projectid && m.category == (int)MachineCategory.NC
                            orderby m.sorting, m.id
                            select q.machine;

            return querylist.ToList();
        }

        public IEnumerable<MachineData> GetNCLevelSectionNames(int id, int projectid)
        {
            var list = new List<int>();
            var query = GetSonLevelSections(id, projectid);

            list.Add(id);
            foreach (var section in query.ToList())
            {
                list.Add(section.id);
            }

            var querylist = from q in _context.siger_project_machine_attribution
                            join m in _context.siger_project_machine on q.machine equals m.id
                            join se in _context.siger_project_level_section on q.station equals se.id
                            join se2 in _context.siger_project_level_section on se.parentid equals se2.id
                            where list.Contains(q.station) && q.status == (int)RowState.Valid &&
                                  m.status == (int)RowState.Valid && se.status == (int)RowState.Valid
                                  && m.projectid == projectid && m.category == (int)MachineCategory.NC orderby m.sorting,m.id
                            select new MachineData
                            {
                                machine_id = m.id,
                                machine_name = m.title,
                                machine_code = m.code,
                                section_id = se.id,
                                lastSectionTitle = se.title,
                                lastSecondSectionTitle = se2.title,
                                sorting = m.sorting
                            };

            return querylist.ToList();
        }

        public IEnumerable<MachineData> GetNCLevelSectionMachines(int id, int projectid)
        {
            var list = new List<int>();
            var query = GetSonLevelSections(id, projectid);

            list.Add(id);
            foreach (var section in query.ToList())
            {
                list.Add(section.id);
            }

            var querylist = from q in _context.siger_project_machine_attribution
                            join m in _context.siger_project_machine on q.machine equals m.id
                            join se in _context.siger_project_level_section on q.station equals se.id
                            where list.Contains(q.station) && q.status == (int)RowState.Valid &&
                                  m.status == (int)RowState.Valid && se.status == (int)RowState.Valid
                                  && m.projectid == projectid
                            select new MachineData
                            {
                                section_id = q.station,
                                machine_id = m.id,
                                machine_name = m.title,
                                machine_code = m.code,
                                lastSectionTitle = se.title,
                                sorting = m.sorting
                            };

            return querylist.OrderBy(q => q.sorting).ToList();
        }

        public IEnumerable<siger_project_level_section> GetNCLevelSections(int id, int projectid)
        {
            var res = new List<siger_project_level_section>();
            var result = _context.siger_project_level
                .Where(f => f.status == RowState.Valid.GetHashCode() && f.projectid == projectid)
                .OrderBy(f => f.id).ToList();
            if (!result.Any())
            {
                return res;
            }
            var list = GetSonLevelSectionsByParentId(id, projectid, result).ToList();
            if (!list.Any())
            {
                return res;
            }
            var removeList = new List<int>();
            foreach (var sections in list)
            {
                if (removeList.Contains(sections.id))
                {
                    continue;
                }
                var deleteList = new List<int> { sections.id };
                var parentIds = new List<int> { sections.id };
                var i = 1;
                foreach (var level in result)
                {
                    if (sections.levelid <= level.id)
                    {
                        if (i == result.Count)
                        {
                            break;
                        }
                        var ids = parentIds;
                        parentIds = list.Where(t => ids.Contains(t.parentid)).Select(t => t.id).ToList();
                        deleteList.AddRange(parentIds);
                    }

                    i++;
                }

                var machineIds = _context.siger_project_machine_attribution.Where(t =>
                    t.status == (int)RowState.Valid && parentIds.Contains(t.station)).Select(t => t.machine).ToList();
                var machines = _context.siger_project_machine.Where(t =>
                    machineIds.Contains(t.id) && t.projectid == projectid && t.status == (int)RowState.Valid &&
                    t.category == (int)MachineCategory.NC);
                if (!machines.Any())
                {
                    removeList.AddRange(deleteList);
                }
            }

            list = list.Where(t => !removeList.Contains(t.id)).ToList();

            return list;
        }

        public IEnumerable<MachineData> GetNCLevelSectionNames(IEnumerable<int> machineIds, int projectid)
        {
            var querylist = from q in _context.siger_project_machine_attribution
                            join m in _context.siger_project_machine on q.machine equals m.id
                            join se in _context.siger_project_level_section on q.station equals se.id
                            join se2 in _context.siger_project_level_section on se.parentid equals se2.id
                            where machineIds.Contains(q.machine) && q.status == (int)RowState.Valid &&
                                  m.status == (int)RowState.Valid && se.status == (int)RowState.Valid
                                  && m.projectid == projectid && m.category == (int)MachineCategory.NC 
                            select new MachineData
                            {
                                machine_id = m.id,
                                machine_name = m.title,
                                machine_code = m.code,
                                section_id = se.id,
                                lastSectionTitle = se.title,
                                lastSecondSectionTitle = se2.title,
                                sorting = m.sorting
                            };

            return querylist.OrderBy(q => q.sorting).ToList();
        }

        public IEnumerable<int> GetNCMachineIdsBySectionIds(IEnumerable<int> sectionIds, int projectid, bool dashboarSlicedata = false)
        {
            if (dashboarSlicedata)
            {
                return from q in _context.siger_project_machine_attribution
                       join m in _context.siger_project_machine on q.machine equals m.id
                       where m.projectid == projectid && q.status == (int)RowState.Valid &&
                             m.status == (int)RowState.Valid && sectionIds.Contains(q.station) && m.category == (int)MachineCategory.NC
                       orderby m.sorting, m.id
                       select q.machine;
            }
            return from q in _context.siger_project_machine_attribution
                   join m in _context.siger_project_machine on q.machine equals m.id
                   where m.projectid == projectid && q.status == (int)RowState.Valid &&
                         m.status == (int)RowState.Valid && sectionIds.Contains(q.station) && m.category == (int)MachineCategory.NC
                   orderby m.sorting, m.id
                   select q.machine;
        }
        /// <summary>
        /// 设备id重新排序
        /// </summary>
        /// <param name="sectionIds">设备id</param>
        /// <param name="projectid">项目id</param>
        /// <returns></returns>
        public IEnumerable<int> GetOrderByNCMachineIds(IEnumerable<int> sectionIds, int projectid)
        {          
            return from  m in _context.siger_project_machine
                   where m.projectid == projectid && m.status == (int)RowState.Valid &&
                   sectionIds.Contains(m.id) && m.category == (int)MachineCategory.NC
                   orderby m.sorting, m.id
                   select m.id;
        }

        private IEnumerable<siger_project_level_section> GetSonLevelSectionsByParentId(int id, int projectid, IEnumerable<siger_project_level> levels)
        {
            var res = new List<siger_project_level_section>();
            if (id > 0)
            {
                var section = _context.siger_project_level_section.FirstOrDefault(t =>
                    t.id == id && t.projectid == projectid && t.status == (int)RowState.Valid);
                if (section == null)
                {
                    return res;
                }
                res.Add(section);
            }

            var parentIds = new List<int> { id };
            foreach (var level in levels)
            {
                var sections = _context.siger_project_level_section.Where(t =>
                    parentIds.Contains(t.parentid) && t.projectid == projectid && t.status == (int)RowState.Valid).ToList();
                if (sections.Any())
                {
                    res.AddRange(sections);
                    parentIds = sections.Select(t => t.id).ToList();
                }
            }

            return res;
        }
        /// <summary>
        /// 最后两级section Desc
        /// </summary>
        /// <param name="machineId"></param>
        /// <param name="projectId"></param>
        /// <returns></returns>
        public string GetLineLevelTitlebByMachine(int machineId, int projectId)
        {
            var query = from lm in _context.siger_project_machine_attribution
                        join lv in _context.siger_project_level_section on lm.station equals lv.id
                        join plv in _context.siger_project_level_section on lv.parentid equals plv.id
                        where lv.projectid == projectId && lv.status == (int)RowState.Valid &&
                        lm.machine == machineId
                        select new LevelTitle
                        {
                            Lastlevel = lv.parentid,
                            LastTitle = lv.title,
                            PreLevel = plv.id,
                            PreTitle = plv.title
                        };
            var data = query.ToArray();
            var obj = data.FirstOrDefault();
            if (obj != null)
            {
                return obj.PreTitle + "-" + obj.LastTitle;
            }
            return machineId.ToString();
        }

        public string GetMachineLocationBySectionId(int sectionId, int projectId)
        {
            var titles = GetLevelSectionTitles(sectionId, projectId);
            return string.Join("-", titles);
        }

        public string GetMachineLocationByMachineId(int machineId, int projectId)
        {
            var section = _context.siger_project_machine_attribution.FirstOrDefault(t =>
                t.machine == machineId && t.status == (int)RowState.Valid);
            if (section == null)
            {
                return "";
            }
            var titles = GetLevelSectionTitles(section.station, projectId);
            return string.Join("-", titles);
        }

        public IEnumerable<siger_project_machine_attribution> GetSonLevelMachineAttribution(int parentId, int projectid)
        {
            var query = from c in _context.siger_project_machine_attribution
                        where c.parent == parentId && c.projectid == projectid && c.status == (int)RowState.Valid
                        select c;

            return query.ToList().Concat(query.ToList().SelectMany(t => GetSonLevelMachineAttribution(t.id, projectid)));
        }
        public IEnumerable<siger_project_machine_attribution> GetParentLevelAttribution(int id, int projectid)
        {
            var query = from c in _context.siger_project_machine_attribution
                        where c.id == id && c.projectid == projectid && c.status == (int)RowState.Valid
                        select c;

            return query.ToList().Concat(query.ToList().SelectMany(t => GetParentLevelAttribution(t.parent, projectid)));
        }

        public IEnumerable<int> GetAttributionMachineIds(int id, int projectid)
        {
            var list = new List<int>();
            var query = GetSonLevelSections(id, projectid).ToList();
            if (!query.Any())
            {
                list.Add(id);
            }
            else
            {
                foreach (var section in query)
                {
                    list.Add(section.id);
                }
            }
            var querylist = from q in _context.siger_project_machine_attribution
                            join m in _context.siger_project_machine on q.machine equals m.id into ma
                            from m in ma.DefaultIfEmpty()
                            where list.Contains(q.station) && q.status == (int)RowState.Valid &&
                                  m.status == (int)RowState.Valid
                                  && m.projectid == projectid && m.attribution < 4
                            orderby m.sorting
                            select q.machine;

            return querylist.ToList();
        }

        public IEnumerable<siger_project_machine> GetAttributionMachineList(int id, int projectid, List<int> attribution)
        {
            var list = new List<int>();
            var query = GetSonLevelSections(id, projectid).ToList();
            if (!query.Any())
            {
                list.Add(id);
            }
            else
            {
                foreach (var section in query)
                {
                    list.Add(section.id);
                }
            }
            var querylist = from q in _context.siger_project_machine_attribution
                            join m in _context.siger_project_machine on q.machine equals m.id into ma
                            from m in ma.DefaultIfEmpty()
                            where list.Contains(q.station) && q.status == (int)RowState.Valid &&
                                  m.status == (int)RowState.Valid
                                  && m.projectid == projectid && attribution.Contains(m.attribution)
                            orderby m.sorting
                            select m;

            return querylist.ToList();
        }

        public IEnumerable<siger_project_machine> GetAttributionMachineList(List<int> sections, int projectid, List<int> attribution)
        {
            var list = new List<int>();
            foreach (var section in sections)
            {
                var query = GetSonLevelSections(section, projectid).ToList();
                if (!query.Any())
                {
                    list.Add(section);
                }
                else
                {
                    foreach (var data in query)
                    {
                        list.Add(data.id);
                    }
                }
            }
            var querylist = from q in _context.siger_project_machine_attribution
                            join m in _context.siger_project_machine on q.machine equals m.id into ma
                            from m in ma.DefaultIfEmpty()
                            where list.Contains(q.station) && q.status == (int)RowState.Valid &&
                                  m.status == (int)RowState.Valid
                                  && m.projectid == projectid && attribution.Contains(m.attribution)
                            orderby m.sorting
                            select m;

            return querylist.ToList();
        }

        public string GetUserNamesByMids(string mids, int projectId)
        {
            var ids = mids.TrimEnd(',').Split(',').Distinct().ToList();
            var users = _context.siger_project_user.Where(q =>
                ids.Contains(q.mid.ToString()) && q.status == (int) RowState.Valid && q.projectid == projectId).Select(m => m.name).ToList();
            if (users.Any())
            {
                return string.Join(",", users);
            }

            return string.Empty;
        }


        public int GetSectionByMachine(int id, int projectid)
        {
            var section = 0;
            var query = _context.siger_project_machine_attribution.Where(q => q.machine == id && q.status == (int)RowState.Valid && q.projectid == projectid).FirstOrDefault();
            if (query!=null)
            {
                section = query.station;
            }
            return section;
        }


        public IEnumerable<CncSliceSate> GetsliceSates(IEnumerable<CncSliceSate> sliceSates, int machineId, int projectid, DateTime start, DateTime end)
        {
            if (sliceSates.Any())
            {
                var query = sliceSates.Where(q => q.MachineID == machineId &&
                             (q.StartTime >= start && q.EndTime <= end) ||
                             (q.StartTime < start && q.EndTime > start && q.EndTime <= end) ||
                             (q.StartTime >= start && q.StartTime < end && q.EndTime > end) ||
                             (q.StartTime < start && q.EndTime > end)).ToList();
                return query;
            }
          
            return null;
        }

        public IEnumerable<ResponseClockRecord> GetMachineuser(string machineCode, int projectid)
        {
            var query = from a in _context.siger_project_machine
                        join b in _context.siger_project_machine_attribution on a.id equals b.machine
                        join c in _context.siger_project_level_section on b.station equals c.id

                        where b.status == (int)RowState.Valid && b.projectid == projectid && a.code == machineCode
                        select new ResponseClockRecord
                        {
                            id = a.id,
                            machinesolution = c.title,
                            machinename = b.name
                        };
            return query.ToList();

        }
    }
}
