﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Mvc;
using Siger.ApiCNC.Result;
using Siger.ApiCommon.Result;
using Siger.ApiCommon.Utilities;
using Siger.Middlelayer.CncRepository.Repositories.Interface;
using Siger.Middlelayer.CncRepository.Request;
using Siger.Middlelayer.CncRepository.Response;
using Siger.Middlelayer.Common;
using Siger.Middlelayer.Common.AppSettings;
using Siger.Middlelayer.Common.Extensions;
using Siger.Middlelayer.Common.Helpers;
using Siger.Middlelayer.Common.ModuleEnum;
using Siger.Middlelayer.Dapper;
using Siger.Middlelayer.Dapper.ResultData;
using Siger.Middlelayer.Dapper.SearchCondition;
using Siger.Middlelayer.Dapper.Utilities;
using Siger.Middlelayer.Dapper.Utilities.Slice;
using Siger.Middlelayer.Log;
using Siger.Middlelayer.Repository.Repositories.Interface;
using Siger.Middlelayer.Repository.Response;
using Siger.Middlelayer.Share.Models;
using Siger.Middlelayer.Utility.ExcelImport;
using Siger.Middlelayer.Utility.Helpers;
using Siger.Middlelayer.Utility.ImportEntities;

namespace Siger.ApiCNC.Controllers
{
    public class ProjectSectionMachineController : BaseController
    {
        private readonly ISigerProjectMachineRepository _machineRepository;
        private readonly IMachineWordingModeRepository _modeRepository;
        private readonly IMachineWordingModeRepository _wordingModeRepository;
        private readonly IMachineStatusTimeSettingRepository _settingRepository;
        private readonly ISigerProjectUserRepository _userRepository;
        private readonly IProductionBeatSetRepository _beatSetRepository;
        private readonly ISigerProjectLevelSectionRepository _levelSectionRepository;
        private readonly IDashboardCombPageConditionTwoRepository _combPageConditionRepository;
        private readonly ISigerProjectConfigFrequencyRepository _configFrequencyRepository;
        private readonly ISigerProjectShiftRepository _sigerProjectShiftRepository;

        public ProjectSectionMachineController(ISigerProjectMachineRepository machineRepository, IDashboardCombPageConditionTwoRepository combPageConditionRepository,
            IMachineWordingModeRepository modeRepository, ISigerProjectLevelSectionRepository levelSectionRepository,
            IMachineWordingModeRepository wordingModeRepository, IMachineStatusTimeSettingRepository settingRepository,
            ISigerProjectUserRepository userRepository, IProductionBeatSetRepository beatSetRepository, ISigerProjectConfigFrequencyRepository sigerProjectConfigFrequencyRepository,
            ISigerProjectShiftRepository sigerProjectShiftRepository)
        {
            _machineRepository = machineRepository;
            _modeRepository = modeRepository;
            _wordingModeRepository = wordingModeRepository;
            _settingRepository = settingRepository;
            _userRepository = userRepository;
            _beatSetRepository = beatSetRepository;
            _levelSectionRepository = levelSectionRepository;
            _combPageConditionRepository = combPageConditionRepository;
            _configFrequencyRepository = sigerProjectConfigFrequencyRepository;
            _sigerProjectShiftRepository = sigerProjectShiftRepository;
        }

        /// <summary>
        /// 设备切片分析，不返回程序号，默认返回前5台设备
        /// </summary>
        /// <param name="sectionId"></param>
        /// <param name="timeSection"></param>
        /// <param name="sectionIdStr"></param>
        /// <returns></returns>
        [HttpGet]
        public async Task<IActionResult> GetMachineCanvas(int sectionId, string timeSection, string sectionIdStr)
        {
            if (string.IsNullOrWhiteSpace(timeSection))
            {
                throw new BadRequestException(RequestEnum.TimeSpanNotNull);
            }

            var times = timeSection.Split(" - ");
            var success = DateTime.TryParse(times[0], out var startTime);
            if (!success)
            {
                throw new ServerException(500146);
            }

            success = DateTime.TryParse(times[1], out var endTime);
            if (!success)
            {
                throw new ServerException(500146);
            }

            var response = new MachineCanvasResult
            {
                startTime = startTime.ToString(ParameterConstant.DateTimeFormat),
                endTime = endTime.ToString(ParameterConstant.DateTimeFormat),
                total_time_span = (endTime - startTime).TotalSeconds
            };

            var sectionIds = string.IsNullOrWhiteSpace(sectionIdStr)
                ? new List<int>()
                : sectionIdStr.Split(',', StringSplitOptions.RemoveEmptyEntries).Select(t => t.ToInt()).ToList();
            var machineIds = sectionIds.Any()
                ? _machineRepository.GetNCMachineIdsBySectionIds(sectionIds, ProjectId).ToList()
                : _machineRepository.GetNCLevelSectionMachineIds(sectionId, ProjectId).ToList();
            var validMachineIds = (sectionId == 0 && !sectionIds.Any()) ? machineIds.Take(5).ToList() : machineIds;
            var validMachines = _machineRepository.GetNCLevelSectionNames(validMachineIds, ProjectId);

            var sliceCanvas =
                await SliceManager.GetMachineSlice(startTime, endTime, machineIds, validMachines, CompanyId, ProjectId);
            response.canvas = sliceCanvas.canvas;
            response.canvas = await ResetCanvas(startTime, validMachineIds, sliceCanvas.canvas);
            return response;
        }

        [HttpGet]
        public async Task<IActionResult> GetMachineYields(int machineId, string timeSection)
        {
            var times = timeSection.Split(" - ");
            var result = new MachineYieldResult
            {
                machineId = machineId
            };

            var beats = _beatSetRepository.GetList(q =>
                q.machineID == machineId && q.projectID == ProjectId && q.status == (int)RowState.Valid).ToList();
            var repository = new ProductRepository(CompanyId, ProjectId);
            var yields = await repository.GetYieldsAsync(times[0], times[1], new List<int>() { machineId });
            if (yields != null)
            {
                var machineYields = yields.Where(m => m.machineID == machineId);
                if (machineYields.Any())
                {
                    var programNos = await YieldMangaer.GetProgramNos(machineYields, times[0], machineId,
                        UnixTimeHelper.GetUnixByShortDate(times[1]));
                    foreach (var program in programNos)
                    {
                        var newProgram = Mapper<ProgramYield, ProgramYield>.Map(program);
                        var end = newProgram.endTime.ToDateTime();
                        var beat = beats.FirstOrDefault(q =>
                            q.machineID == program.machineId && q.process_number == program.programNo &&
                            q.start_time <= end && q.end_time >= end);
                        if (beat != null)
                        {
                            newProgram.yield = newProgram.yield * beat.yieldrate;
                        }

                        newProgram.startTime = UnixTimeHelper.GetUnixByShortDate(program.startTime).ToStr();
                        newProgram.endTime = UnixTimeHelper.GetUnixByShortDate(program.endTime).ToStr();

                        result.programNos.Add(newProgram);
                    }
                }
            }

            return result;
        }

        private async Task<IList<Canvas>> ResetCanvas(DateTime startTime, IEnumerable<int> validMachineIds,
            IList<Canvas> canvas)
        {
            var settings = _settingRepository.GetMachineSettings(validMachineIds, ProjectId);
            var unixStartTime = UnixTimeHelper.ConvertDataTimeLong(startTime);

            var parentWorkMode = string.Empty;
            var parentWorkModeId = 0;
            return await Task.Run(() =>
            {
                foreach (var canva in canvas)
                {
                    var listMsh = new List<MachineStatusHold>();
                    foreach (var hold in canva.list_msh)
                    {
                        if (hold.workModeId > 0)
                        {
                            var workMode = _wordingModeRepository.Get(t =>
                                t.id == hold.workModeId && t.projectid == ProjectId &&
                                t.status == (int)RowState.Valid);
                            if (workMode != null)
                            {
                                var parentMode = _wordingModeRepository.Get(t =>
                                    t.id == workMode.parentid && t.projectid == ProjectId &&
                                    t.status == (int)RowState.Valid);
                                parentWorkMode = parentMode?.workingmodename ?? "";
                                parentWorkModeId = parentMode?.id ?? 0;
                            }
                        }

                        listMsh.Add(new MachineStatusHold
                        {
                            end_time = hold.end_time,
                            isException = CheckIsException(hold.machine_status, hold.time_span, canva.machineID,
                                settings),
                            machine_status = hold.machine_status,
                            start_time = hold.start_time,
                            time_span = hold.time_span,
                            workMode = hold.workMode.ToStr(),
                            workModeId = hold.workModeId,
                            workModeParentId = parentWorkModeId,
                            workModeParent = parentWorkMode,
                            remark = string.IsNullOrWhiteSpace(hold.remark)
                                ? " "
                                : (hold.remark.StartsWith(" ") ? hold.remark : " " + hold.remark),
                            userId = GetUserIdByName(hold.userName)
                        });
                    }

                    var minListTime = listMsh.Min(q => q.start_time);
                    if (minListTime - unixStartTime > 5 * 60)
                    {
                        listMsh.Insert(0, new MachineStatusHold
                        {
                            end_time = minListTime,
                            machine_status = 0,
                            start_time = (int)unixStartTime,
                            time_span = minListTime - (int)unixStartTime,
                            isFake = 1
                        });
                    }

                    canva.list_msh = listMsh;
                }

                return canvas;
            });
        }

        /// <summary>
        /// 设备切片,不包含程序号
        /// </summary>
        /// <param name="machineIds"></param>
        /// <param name="timeSection"></param>
        /// <returns></returns>
        [HttpGet]
        public async Task<IActionResult> MachineCanvasParetoInfo(List<int> machineIds, string timeSection)
        {
            var result = CheckParameter(timeSection);
            var time = result.endTime.ToDateTime();
            var dtStart = result.startTime.ToDateTime();
            var dtEnd = time > DateTime.Now ? DateTime.Now : time;

            var validMachines = _machineRepository.GetNCLevelSectionNames(machineIds, ProjectId);

            var canvas =
                await SliceManager.GetMachineSliceNew(dtStart, dtEnd, machineIds, validMachines, CompanyId, ProjectId);
            result.canvas = canvas.canvas;
            result.pareto = canvas.pareto;
            result.endTime = dtEnd.ToString(ParameterConstant.DateTimeFormat);
            return result;
        }

        /// <summary>
        /// 设备切片分析，包含程序号
        /// </summary>
        /// <param name="sectionId"></param>
        /// <param name="timeSection"></param>
        /// <param name="sectionIdStr"></param>
        /// <returns></returns>
        [HttpGet]
        public async Task<IActionResult> MachineCanvasPrograms(int sectionId, string timeSection, string sectionIdStr)
        {
            var response = new Result.MachineCanvasParetoInfoResult();
            var sectionIds = string.IsNullOrWhiteSpace(sectionIdStr)
                ? new List<int>()
                : sectionIdStr.Split(',', StringSplitOptions.RemoveEmptyEntries).Select(t => t.ToInt()).ToList();
            var machineIds = sectionIds.Any()
                //判断设备
                ? _machineRepository.GetNCMachineIdsBySectionIds(sectionIds, ProjectId, true).ToList()
                : _machineRepository.GetNCLevelSectionMachineIds(sectionId, ProjectId).ToList();
            //var machineIds = sectionIds.Any()
            //    //判断设备
            //    ? sectionIds
            //    : _machineRepository.GetNCLevelSectionMachineIds(sectionId, ProjectId).ToList();
            machineIds = _machineRepository.GetOrderByNCMachineIds(machineIds, ProjectId).ToList();
            var machineIDs = (sectionId == 0 && !sectionIds.Any()) ? machineIds.Take(15).ToList() : machineIds;

            var beats = _beatSetRepository.GetList(q =>
                    machineIDs.Contains(q.machineID) && q.projectID == ProjectId && q.status == (int)RowState.Valid)
                .ToList();
            if (await MachineCanvasParetoInfo(machineIDs, timeSection) is Result.MachineCanvasParetoInfoResult result)
            {
                response.endTime = result.endTime;
                response.pareto = result.pareto;
                response.startTime = result.startTime;
                response.total_time_span = result.total_time_span;

                //异常设置
                var settings = _settingRepository.GetMachineSettings(machineIDs, ProjectId);

                //获取程序号
                var minStartTime = DateTime.Parse(result.startTime);
                var unixStartTime = UnixTimeHelper.ConvertDataTimeLong(minStartTime);
                var repository = new ProductRepository(CompanyId, ProjectId);
                var yields = await repository.GetYieldsAsync(response.startTime, response.endTime, machineIDs);
                if (yields != null)
                {
                    var allprogramnos = GetColorByProgram(yields.Select(m => m.programCode).Distinct().ToList());
                    var canvas = result.canvas;
                    foreach (var canva in canvas)
                    {
                        var listMsh = new List<MachineStatusHold>();
                        foreach (var hold in canva.list_msh)
                        {
                            var parentWorkMode = string.Empty;
                            var parentWorkModeId = 0;
                            if (hold.workModeId > 0)
                            {
                                var workMode = _wordingModeRepository.Get(t =>
                                    t.id == hold.workModeId && t.projectid == ProjectId &&
                                    t.status == (int)RowState.Valid);
                                if (workMode != null)
                                {
                                    var parentMode = _wordingModeRepository.Get(t =>
                                        t.id == workMode.parentid && t.projectid == ProjectId &&
                                        t.status == (int)RowState.Valid);
                                    parentWorkMode = parentMode?.workingmodename ?? "";
                                    parentWorkModeId = parentMode?.id ?? 0;
                                }
                            }

                            listMsh.Add(new MachineStatusHold
                            {
                                end_time = hold.end_time,
                                isException = CheckIsException(hold.machine_status, hold.time_span, canva.machineID,
                                    settings),
                                machine_status = hold.machine_status,
                                start_time = hold.start_time,
                                time_span = hold.time_span,
                                workMode = hold.workMode.ToStr(),
                                workModeId = hold.workModeId,
                                workModeParentId = parentWorkModeId,
                                workModeParent = parentWorkMode.ToStr(),
                                remark = string.IsNullOrWhiteSpace(hold.remark)
                                    ? " "
                                    : (hold.remark.StartsWith(" ") ? hold.remark : " " + hold.remark),
                                userId = GetUserIdByName(hold.userName)
                            });
                        }

                        var minListTime = listMsh.Min(q => q.start_time);
                        if (minListTime - unixStartTime > 5 * 60)
                        {
                            listMsh.Insert(0, new MachineStatusHold
                            {
                                end_time = minListTime,
                                machine_status = 0,
                                start_time = (int)unixStartTime,
                                time_span = minListTime - (int)unixStartTime,
                                isFake = 1
                            });
                        }

                        canva.list_msh = listMsh;

                        var machineYields = yields.Where(m => m.machineID == canva.machineID);
                        var maxEndTime = canva.list_msh.Max(m => m.end_time);
                        if (machineYields.Any())
                        {
                            var programNos = await YieldMangaer.GetProgramNos(machineYields,
                                minStartTime.ToString(ParameterConstant.DateTimeFormat), canva.machineID, maxEndTime);
                            //程序号
                            foreach (var program in programNos)
                            {
                                var newProgram = Mapper<ProgramYield, ProgramYield>.Map(program);
                                var end = newProgram.endTime.ToDateTime();
                                var beat = beats.FirstOrDefault(q =>
                                    q.machineID == program.machineId && q.process_number == program.programNo &&
                                    q.start_time <= end && q.end_time >= end);
                                if (beat != null)
                                {
                                    newProgram.yield = newProgram.yield * beat.yieldrate;
                                }

                                var color = allprogramnos.FirstOrDefault(q => q.programNo == program.programNo);
                                if (color != null)
                                {
                                    newProgram.color = color.color;
                                }

                                canva.programNos.Add(newProgram);
                            }
                        }
                    }
                    response.canvas = canvas;
                }
            }
            else
            {
                return response;
            }

            return response;
        }

        /// <summary>
        /// 设备切片分析，包含程序号 大屏专用
        /// </summary>
        /// <param name="sectionId"></param>
        /// <param name="timeSection"></param>
        /// <param name="sectionIdStr"></param>
        /// <returns></returns>
        [HttpGet]
        public async Task<IActionResult> MachineCanvasProgramsDP(int sectionId, string timeSection, string sectionIdStr)
        {
            var response = new Result.MachineCanvasParetoInfoResult();
            var sectionIds = string.IsNullOrWhiteSpace(sectionIdStr)
                ? new List<int>()
                : sectionIdStr.Split(',', StringSplitOptions.RemoveEmptyEntries).Select(t => t.ToInt()).ToList();
            //var machineIds = sectionIds.Any()
            //    //判断设备
            //    ? _machineRepository.GetNCMachineIdsBySectionIds(sectionIds, ProjectId, true).ToList()
            //    : _machineRepository.GetNCLevelSectionMachineIds(sectionId, ProjectId).ToList();
            var machineIds = sectionIds.Any()
                //判断设备
                ? sectionIds
                : _machineRepository.GetNCLevelSectionMachineIds(sectionId, ProjectId).ToList();
            machineIds = _machineRepository.GetOrderByNCMachineIds(machineIds, ProjectId).ToList();
            var machineIDs = (sectionId == 0 && !sectionIds.Any()) ? machineIds.Take(15).ToList() : machineIds;

            var beats = _beatSetRepository.GetList(q =>
                    machineIDs.Contains(q.machineID) && q.projectID == ProjectId && q.status == (int)RowState.Valid)
                .ToList();
            if (await MachineCanvasParetoInfo(machineIDs, timeSection) is Result.MachineCanvasParetoInfoResult result)
            {
                response.endTime = result.endTime;
                response.pareto = result.pareto;
                response.startTime = result.startTime;
                response.total_time_span = result.total_time_span;

                //异常设置
                var settings = _settingRepository.GetMachineSettings(machineIDs, ProjectId);

                //获取程序号
                var minStartTime = DateTime.Parse(result.startTime);
                var unixStartTime = UnixTimeHelper.ConvertDataTimeLong(minStartTime);
                var repository = new ProductRepository(CompanyId, ProjectId);
                var yields = await repository.GetYieldsAsync(response.startTime, response.endTime, machineIDs);
                if (yields != null)
                {
                    var allprogramnos = GetColorByProgram(yields.Select(m => m.programCode).Distinct().ToList());
                    var canvas = result.canvas;
                    foreach (var canva in canvas)
                    {
                        var listMsh = new List<MachineStatusHold>();
                        foreach (var hold in canva.list_msh)
                        {
                            var parentWorkMode = string.Empty;
                            var parentWorkModeId = 0;
                            if (hold.workModeId > 0)
                            {
                                var workMode = _wordingModeRepository.Get(t =>
                                    t.id == hold.workModeId && t.projectid == ProjectId &&
                                    t.status == (int)RowState.Valid);
                                if (workMode != null)
                                {
                                    var parentMode = _wordingModeRepository.Get(t =>
                                        t.id == workMode.parentid && t.projectid == ProjectId &&
                                        t.status == (int)RowState.Valid);
                                    parentWorkMode = parentMode?.workingmodename ?? "";
                                    parentWorkModeId = parentMode?.id ?? 0;
                                }
                            }

                            listMsh.Add(new MachineStatusHold
                            {
                                end_time = hold.end_time,
                                isException = CheckIsException(hold.machine_status, hold.time_span, canva.machineID,
                                    settings),
                                machine_status = hold.machine_status,
                                start_time = hold.start_time,
                                time_span = hold.time_span,
                                workMode = hold.workMode.ToStr(),
                                workModeId = hold.workModeId,
                                workModeParentId = parentWorkModeId,
                                workModeParent = parentWorkMode.ToStr(),
                                remark = string.IsNullOrWhiteSpace(hold.remark)
                                    ? " "
                                    : (hold.remark.StartsWith(" ") ? hold.remark : " " + hold.remark),
                                userId = GetUserIdByName(hold.userName)
                            });
                        }

                        var minListTime = listMsh.Min(q => q.start_time);
                        if (minListTime - unixStartTime > 5 * 60)
                        {
                            listMsh.Insert(0, new MachineStatusHold
                            {
                                end_time = minListTime,
                                machine_status = 0,
                                start_time = (int)unixStartTime,
                                time_span = minListTime - (int)unixStartTime,
                                isFake = 1
                            });
                        }

                        canva.list_msh = listMsh;

                        var machineYields = yields.Where(m => m.machineID == canva.machineID);
                        var maxEndTime = canva.list_msh.Max(m => m.end_time);
                        if (machineYields.Any())
                        {
                            var programNos = await YieldMangaer.GetProgramNos(machineYields,
                                minStartTime.ToString(ParameterConstant.DateTimeFormat), canva.machineID, maxEndTime);
                            //程序号
                            foreach (var program in programNos)
                            {
                                var newProgram = Mapper<ProgramYield, ProgramYield>.Map(program);
                                var end = newProgram.endTime.ToDateTime();
                                var beat = beats.FirstOrDefault(q =>
                                    q.machineID == program.machineId && q.process_number == program.programNo &&
                                    q.start_time <= end && q.end_time >= end);
                                if (beat != null)
                                {
                                    newProgram.yield = newProgram.yield * beat.yieldrate;
                                }

                                var color = allprogramnos.FirstOrDefault(q => q.programNo == program.programNo);
                                if (color != null)
                                {
                                    newProgram.color = color.color;
                                }

                                canva.programNos.Add(newProgram);
                            }
                        }
                    }
                    response.canvas = canvas;
                }
            }
            else
            {
                return response;
            }

            return response;
        }

        private static IList<ProgramColor> GetColorByProgram(IReadOnlyCollection<string> programNos)
        {
            var colors = ColorHelper.Colors;
            if (programNos.Count > colors.Count)
            {
                int differNumber = programNos.Count - colors.Count;
                for (var i = 0; i < differNumber; i++)
                {
                    colors.Add(ColorHelper.RandomColor());
                }
            }

            return programNos.Select((program, i) => new ProgramColor
            {
                color = colors[i],
                programNo = program
            }).ToList();
        }

        private int GetUserIdByName(string name)
        {
            if (string.IsNullOrWhiteSpace(name))
            {
                return 0;
            }

            var user = _userRepository.Get(q => q.name == name && q.projectid == ProjectId);
            if (user != null)
            {
                return user.mid;
            }

            return 0;
        }

        //验证是否异常
        private int CheckIsException(int status, double timeSpan, int machineId,
            IEnumerable<ResponseGetMachineSetting> settings)
        {
            if (!settings.Any())
            {
                return 0;
            }

            //调试，空闲，故障
            if (status == 2 || status == 3 || status == 4)
            {
                var setting = settings.FirstOrDefault(q => q.machineId == machineId);
                if (setting != null)
                {
                    switch (status)
                    {
                        case 2:
                            if (setting.debugging * 60 < timeSpan)
                            {
                                return 1;
                            }

                            break;
                        case 3:
                            if (setting.free * 60 < timeSpan)
                            {
                                return 1;
                            }

                            break;
                        case 4:
                            if (setting.fault * 60 < timeSpan)
                            {
                                return 1;
                            }

                            break;
                        default:
                            return 0;
                    }
                }
            }

            return 0;
        }

        private Result.MachineCanvasParetoInfoResult CheckParameter(string timeSection)
        {
            if (string.IsNullOrWhiteSpace(timeSection))
            {
                throw new BadRequestException(RequestEnum.TimeSpanNotNull);
            }

            var times = timeSection.Split(" - ");
            var success = DateTime.TryParse(times[0], out var startTime);
            if (!success)
            {
                throw new ServerException(500146);
            }

            success = DateTime.TryParse(times[1], out var endTime);
            if (!success)
            {
                throw new ServerException(500146);
            }

            //if (endTime > DateTime.Now)
            //{
            //    endTime = DateTime.Now;
            //}
            //if (startTime > endTime)
            //{
            //    throw new ServerException(100218);
            //}
            var result = new Result.MachineCanvasParetoInfoResult
            {
                startTime = startTime.ToString(ParameterConstant.DateTimeFormat),
                endTime = endTime.ToString(ParameterConstant.DateTimeFormat),
                total_time_span = (endTime - startTime).TotalSeconds
            };
            return result;
        }

        /// <summary>
        /// 设备切片分析
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<IActionResult> MachineCanvasDataInfo([FromBody] RequestGetMachineCanvasDataInfo request)
        {
            //xtype - 0:时间 1：层级
            //x_ID - 0:天 1：周 2：月 xtype=1时，为levelid
            if (!DateTime.TryParse(request.startTime, out var dtStart))
            {
                dtStart = DateTime.Now.AddDays(-14);
            }

            if (!DateTime.TryParse(request.endTime, out var dtEnd))
            {
                dtEnd = DateTime.Now;
            }

            if (dtStart > dtEnd)
            {
                throw new ServerException(100158);
            }

            if (dtStart == dtEnd)
            {
                dtStart = DateTime.Parse(dtStart.ToString(ParameterConstant.DateFromZero));
                dtEnd = DateTime.Parse(dtEnd.ToString(ParameterConstant.DateToEnd));
            }

            dtEnd = dtEnd > DateTime.Now ? DateTime.Now : dtEnd;

            var machinIds = new List<int>();
            var sectionIds = string.IsNullOrWhiteSpace(request.sectionIdStr)
                ? new List<int>()
                : request.sectionIdStr.Split(',', StringSplitOptions.RemoveEmptyEntries).Select(t => t.ToInt())
                    .ToList();
            machinIds = sectionIds.Any()
                ? _machineRepository.GetNCMachineIdsBySectionIds(sectionIds, ProjectId).ToList()
                : _machineRepository.GetNCLevelSectionMachineIds(request.sectionId, ProjectId).ToList();

            var responses = new List<MachineCanvasDataInfo>();
            var patchId = request.x_id.ToInt();
            if (request.xtype == 0)
            {
                var type = 0;
                if (patchId == 1)
                {
                    type = 11;
                }
                else if (patchId == 2)
                {
                    type = 12;
                }
                else
                {
                    type = patchId;
                }
                var interal = DateTimeHelper.GetInteral(patchId);
                var dates = DateTimeHelper.GetDateTimes(dtStart, dtEnd, type);
                var xData = DateTimeHelper.GetStrDateTimes(dtStart, dtEnd, type);
                var count = 0;
                foreach (var date in dates)
                {
                    DateTime start;
                    DateTime end;
                    string dateTime;
                    if (patchId == 0)
                    {
                        start = date;
                        end = date.AddDays(interal).AddSeconds(-1) > DateTime.Now
                            ? DateTime.Now
                            : date.AddDays(interal).AddSeconds(-1);
                        dateTime = end.ToString(ParameterConstant.DateFormat);
                    }
                    else if (patchId == 1)
                    {
                        //start = date;
                        //end = date.AddDays(interal).AddSeconds(-1) > DateTime.Now
                        //    ? DateTime.Now
                        //    : date.AddDays(interal).AddSeconds(-1);
                        dateTime = xData[count];
                        var weekStart = DateTimeHelper.GetWeekRangeStart(dateTime);
                        var weekEnd = DateTimeHelper.GetWeekRangeEnd(dateTime);
                        start = dtStart > weekStart ? dtStart : weekStart;
                        end = dtEnd < weekEnd ? dtEnd.AddDays(1).AddSeconds(-1) : weekEnd.AddDays(1).AddSeconds(-1);
                    }
                    else
                    {
                        start = dates[count];
                        var dts = start.AddMonths(1);
                        dts = dts.AddDays(0 - dts.Day).AddDays(1).AddSeconds(-1);
                        end = dts > dtEnd ? dtEnd.AddDays(1).AddSeconds(-1) : dts;//到月底
                        dateTime = end.ToString("yyyy-MM");
                    }

                    responses.Add(await SliceManager.GetMachineSlice(start, end, machinIds, CompanyId, ProjectId,
                        dateTime, 2));
                    count++;
                }
            }

            if (request.xtype == 1)
            {
                try
                {

                    var sections = sectionIds.Any()
                        ? _machineRepository.GetLevelSectionsBySectionIds(sectionIds, ProjectId)
                        : _machineRepository.GetNCLevelSections(request.sectionId, ProjectId);
                    var levels = sections.Where(q => q.levelid == patchId).ToList();
                    if (levels.Any())
                    {
                        foreach (var levelSection in levels)
                        {
                            var machineIds = _machineRepository.GetNCLevelSectionMachineIds(levelSection.id, ProjectId);
                            var dateTime = levelSection.title;
                            responses.Add(await SliceManager.GetMachineSlice(dtStart, dtEnd, machineIds, CompanyId,
                                ProjectId, dateTime, 2));
                        }
                    }
                    else //最后一级
                    {
                        var machine = _machineRepository.GetNCMachineBySectionId(request.sectionId, ProjectId);
                        var dateTime = machine.name;
                        responses.Add(await SliceManager.GetMachineSlice(dtStart, dtEnd, new List<int> { machine.id },
                            CompanyId, ProjectId, dateTime, 2));
                    }
                }
                catch
                {
                    throw new BadRequestException(CommonEnum.RecordNotFound);
                }
            }

            if (request.toexcel == 0)
            {
                return new ObjectResult(responses);
            }

            return ExportBarChart(responses);
        }

        /// <summary>
        /// 设备切片分析cl
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<IActionResult> MachineCanvasDataInfoCL([FromBody] RequestGetMachineCanvasDataInfo request)
        {
            //xtype - 0:时间 1：层级 2:班次
            //x_ID - 0:天 1：周 2：月 xtype=1时，为levelid
            var dtStrStart = request.startTime;
            var dtStrEnd = request.endTime;

            DateTime.TryParse((dtStrStart), out DateTime dtStart);
            DateTime.TryParse((dtStrEnd), out DateTime dtEnd);
            var timeDifference = 0;
            var totalCount = 0.00;
            var total = 0.00;
            if (dtStart > dtEnd)
            {
                throw new ServerException(100158);
            }

            dtEnd = dtEnd > DateTime.Now ? DateTime.Now : dtEnd;

            var machinIds = new List<int>();
            var sectionIds = string.IsNullOrWhiteSpace(request.sectionIdStr)
                ? new List<int>()
                : request.sectionIdStr.Split(',', StringSplitOptions.RemoveEmptyEntries).Select(t => t.ToInt())
                    .ToList();
            machinIds = sectionIds.Any()
                ? _machineRepository.GetNCMachineIdsBySectionIds(sectionIds, ProjectId).ToList()
                : _machineRepository.GetNCLevelSectionMachineIds(request.sectionId, ProjectId).ToList();

            var responses = new List<MachineCanvasDataInfo>();
            var responsesList = new List<MachineCanvasDataInfo>();
            var patchId = request.x_id.ToInt();
            if (request.xtype == 0)
            {
                var type = 0;
                if (patchId == 1)
                {
                    type = 11;
                }
                else if (patchId == 2)
                {
                    type = 12;
                }
                else
                {
                    type = patchId;
                }
                var interal = DateTimeHelper.GetInteral(patchId);
                var dates = DateTimeHelper.GetDateTimes(dtStart, dtEnd, type);
                var xData = DateTimeHelper.GetStrDateTimes(dtStart, dtEnd, type);
                var count = 0;
                foreach (var date in dates)
                {
                    DateTime start;
                    DateTime end;
                    string dateTime;
                    if (patchId == 0)
                    {
                        var startTime = request.startTime;
                        var endTime = request.endTime;
                        startTime = dtStart.ToString(ParameterConstant.TimeFormat);
                        endTime = dtEnd.ToString(ParameterConstant.TimeFormat);

                        DateTime.TryParse(date.ToString(ParameterConstant.DateFormat) + " " + startTime, out start);

                        end = date.AddDays(interal).AddSeconds(-1) > DateTime.Now
                            ? DateTime.Now
                            : date.AddDays(interal).AddSeconds(-1);
                        if (end > dtEnd)
                        {
                            end = dtEnd;
                        }

                        if (request.frequency != 0)
                        {
                            var timeList = new List<ResponseStartEnd>();

                            var result = _sigerProjectShiftRepository.Get(s => s.projectid == ProjectId && s.status == (int)RowState.Valid && s.id == request.frequency);
                            timeList.Add(new ResponseStartEnd
                            {
                                id = result.id,
                                title = result.title,
                                start = result.start_time,
                                end = result.end_time
                            });
                            foreach (var classDate in timeList)
                            {
                                dateTime = end.ToString(ParameterConstant.DateFormat);
                                if (classDate.end > classDate.start)
                                {
                                    var startTimeClass = UnixTimeHelper.GetTimeByUnix(classDate.start);
                                    var endTimeClass = UnixTimeHelper.GetTimeByUnix(classDate.end);

                                    DateTime.TryParse(start.ToString(ParameterConstant.DateFormat) + " " + startTimeClass, out start);
                                    DateTime.TryParse(end.ToString(ParameterConstant.DateFormat) + " " + endTimeClass, out end); ;
                                    timeDifference = (int)(end - start).TotalHours;
                                    responsesList.Add(await SliceManager.GetMachineSlice(start, end, machinIds, CompanyId, ProjectId,
                                    dateTime, 2));
                                    count++;
                                }
                                else
                                {
                                    var startTimeClass = UnixTimeHelper.GetTimeByUnix(classDate.start);
                                    var endTimeClass = UnixTimeHelper.GetTimeByUnix(classDate.end);

                                    DateTime.TryParse(start.AddDays(1).ToString(ParameterConstant.DateFormat) + " " + endTimeClass, out end);
                                    DateTime.TryParse(start.ToString(ParameterConstant.DateFormat) + " " + startTimeClass, out start);
                                    timeDifference = (int)(end - start).TotalHours;
                                    responsesList.Add(await SliceManager.GetMachineSlice(start, end, machinIds, CompanyId, ProjectId,
                                    dateTime, 2));
                                    count++;
                                }
                            }
                        }
                        else
                        {
                            dateTime = end.ToString(ParameterConstant.DateFormat);
                            responsesList.Add(await SliceManager.GetMachineSlice(start, end, machinIds, CompanyId, ProjectId,
                            dateTime, 2));
                            count++;
                        }

                    }
                    else if (patchId == 1)
                    {
                        if (request.frequency != 0)
                        {

                            dateTime = xData[count];
                            var weekStart = DateTimeHelper.GetWeekRangeStart(dateTime);
                            var weekEnd = DateTimeHelper.GetWeekRangeEnd(dateTime);
                            start = dtStart > weekStart ? dtStart : weekStart;
                            end = dtEnd < weekEnd ? dtEnd.AddDays(1).AddSeconds(-1) : weekEnd.AddDays(1).AddSeconds(-1);

                            var datesList = DateTimeHelper.GetDateTimes(start, end, 0);
                            foreach (var itemData in datesList)
                            {

                                var timeList = new List<ResponseStartEnd>();

                                var result = _sigerProjectShiftRepository.Get(s => s.projectid == ProjectId && s.status == (int)RowState.Valid && s.id == request.frequency);
                                timeList.Add(new ResponseStartEnd
                                {
                                    id = result.id,
                                    title = result.title,
                                    start = result.start_time,
                                    end = result.end_time
                                });

                                foreach (var classDate in timeList)
                                {
                                    DateTime startClass;
                                    DateTime endClass;

                                    startClass = itemData;
                                    endClass = itemData.AddDays(1).AddSeconds(-1) > DateTime.Now
                                        ? DateTime.Now
                                        : itemData.AddDays(1).AddSeconds(-1);

                                    if (startClass <= endClass)
                                    {
                                        if (classDate.end > classDate.start)
                                        {
                                            var startTimeClass = UnixTimeHelper.GetTimeByUnix(classDate.start);
                                            var endTimeClass = UnixTimeHelper.GetTimeByUnix(classDate.end);

                                            DateTime.TryParse(startClass.ToString(ParameterConstant.DateFormat) + " " + startTimeClass, out startClass);
                                            DateTime.TryParse(endClass.ToString(ParameterConstant.DateFormat) + " " + endTimeClass, out endClass); ;

                                            timeDifference = (int)(endClass - startClass).TotalHours;
                                            DateTime date1Start = DateTime.Parse(datesList.First().ToString(ParameterConstant.DateFormat) + " " + startTimeClass);
                                            DateTime date1End = DateTime.Parse(datesList.Last().ToString(ParameterConstant.DateFormat) + " " + endTimeClass);
                                            totalCount = (endClass - startClass).TotalSeconds * machinIds.Count();
                                            total = totalCount <= 0 ? 1 : totalCount;

                                            responses.Add(await SliceManager.GetMachineSlice(startClass, endClass, machinIds, CompanyId, ProjectId,
                                            dateTime, 2));
                                        }
                                        else
                                        {
                                            var startTimeClass = UnixTimeHelper.GetTimeByUnix(classDate.start);
                                            var endTimeClass = UnixTimeHelper.GetTimeByUnix(classDate.end);

                                            DateTime.TryParse(startClass.AddDays(1).ToString(ParameterConstant.DateFormat) + " " + endTimeClass, out endClass);
                                            DateTime.TryParse(startClass.ToString(ParameterConstant.DateFormat) + " " + startTimeClass, out startClass);

                                            timeDifference = (int)(endClass - startClass).TotalHours;

                                            DateTime date1Start = DateTime.Parse(datesList.First().ToString(ParameterConstant.DateFormat) + " " + startTimeClass);
                                            DateTime date1End = DateTime.Parse(datesList.Last().ToString(ParameterConstant.DateFormat) + " " + endTimeClass);
                                            totalCount = (endClass - startClass).TotalSeconds * machinIds.Count();
                                            total = totalCount <= 0 ? 1 : totalCount;

                                            responses.Add(await SliceManager.GetMachineSlice(startClass, endClass, machinIds, CompanyId, ProjectId,
                                            dateTime, 2));
                                        }
                                    }
                                }
                            }
                            var breakMathine = responses.Count(s => s.data.num1 == 100 && s.data.num2 == 0 && s.data.num3 == 0 && s.data.num4 == 0 && s.data.num5 == 0 && s.data.num6 == 0) == responses.Count ? 0 : responses.Count(s => s.data.num1 == 100 && s.data.num2 == 0 && s.data.num3 == 0 && s.data.num4 == 0 && s.data.num5 == 0 && s.data.num6 == 0);
                            count++;
                            var dataList = new MachineSateCount
                            {
                                hour1 = Math.Round((responses.Sum(s => s.data.hour1)), 2),
                                hour2 = Math.Round((responses.Sum(s => s.data.hour2)), 2),
                                hour3 = Math.Round((responses.Sum(s => s.data.hour3)), 2),
                                hour4 = Math.Round((responses.Sum(s => s.data.hour4)), 2),
                                hour5 = Math.Round((responses.Sum(s => s.data.hour5)), 2),
                                hour6 = Math.Round((responses.Sum(s => s.data.hour6)), 2),
                                num1 = responses.Sum(s => s.data.hour1) == 0 ? 100 : Math.Round((responses.Sum(s => s.data.hour1)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                                num2 = responses.Sum(s => s.data.hour2) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour2)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                                num3 = responses.Sum(s => s.data.hour3) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour3)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                                num4 = responses.Sum(s => s.data.hour4) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour4)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                                num5 = responses.Sum(s => s.data.hour5) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour5)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                                num6 = responses.Sum(s => s.data.hour6) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour6)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                                datetime = responses.First().datetime
                            };

                            responsesList.Add(new MachineCanvasDataInfo() { data = dataList, datetime = responses.First().datetime });
                            responses.Clear();

                        }
                        else
                        {
                            dateTime = xData[count];
                            var weekStart = DateTimeHelper.GetWeekRangeStart(dateTime);
                            var weekEnd = DateTimeHelper.GetWeekRangeEnd(dateTime);
                            start = dtStart > weekStart ? dtStart : weekStart;
                            end = dtEnd < weekEnd ? dtEnd.AddDays(1).AddSeconds(-1) : weekEnd.AddDays(1).AddSeconds(-1);

                            responsesList.Add(await SliceManager.GetMachineSlice(start, end, machinIds, CompanyId, ProjectId,
                            dateTime, 2));
                            count++;
                        }
                    }
                    else
                    {
                        if (request.frequency != 0)
                        {
                            start = dates[count];
                            var dts = start.AddMonths(1);
                            dts = dts.AddDays(0 - dts.Day).AddDays(1).AddSeconds(-1);
                            end = dts > dtEnd ? dtEnd.AddDays(1).AddSeconds(-1) : dts;//到月底
                            dateTime = end.ToString("yyyy-MM");

                            var datesList = DateTimeHelper.GetDateTimes(start, end, 0);
                            foreach (var itemData in datesList)
                            {
                                var timeList = new List<ResponseStartEnd>();

                                var result = _sigerProjectShiftRepository.Get(s => s.projectid == ProjectId && s.status == (int)RowState.Valid && s.id == request.frequency);
                                timeList.Add(new ResponseStartEnd
                                {
                                    id = result.id,
                                    title = result.title,
                                    start = result.start_time,
                                    end = result.end_time
                                });
                                foreach (var classDate in timeList)
                                {
                                    DateTime startClass;
                                    DateTime endClass;

                                    startClass = itemData;
                                    endClass = itemData.AddDays(1).AddSeconds(-1) > DateTime.Now
                                        ? DateTime.Now
                                        : itemData.AddDays(1).AddSeconds(-1);


                                    if (startClass <= endClass)
                                    {
                                        if (classDate.end > classDate.start)
                                        {
                                            var startTimeClass = UnixTimeHelper.GetTimeByUnix(classDate.start);
                                            var endTimeClass = UnixTimeHelper.GetTimeByUnix(classDate.end);

                                            DateTime.TryParse(startClass.ToString(ParameterConstant.DateFormat) + " " + startTimeClass, out startClass);
                                            DateTime.TryParse(endClass.ToString(ParameterConstant.DateFormat) + " " + endTimeClass, out endClass); ;
                                            timeDifference = (int)(endClass - startClass).TotalHours;

                                            responses.Add(await SliceManager.GetMachineSlice(startClass, endClass, machinIds, CompanyId, ProjectId,
                                            dateTime, 2));
                                        }
                                        else
                                        {
                                            var startTimeClass = UnixTimeHelper.GetTimeByUnix(classDate.start);
                                            var endTimeClass = UnixTimeHelper.GetTimeByUnix(classDate.end);

                                            DateTime.TryParse(startClass.AddDays(1).ToString(ParameterConstant.DateFormat) + " " + endTimeClass, out endClass);
                                            DateTime.TryParse(startClass.ToString(ParameterConstant.DateFormat) + " " + startTimeClass, out startClass);

                                            timeDifference = (int)(endClass - startClass).TotalHours;
                                            responses.Add(await SliceManager.GetMachineSlice(startClass, endClass, machinIds, CompanyId, ProjectId,
                                            dateTime, 2));
                                        }
                                    }
                                }
                            }
                            var breakMathine = responses.Count(s => s.data.num1 == 100 && s.data.num2 == 0 && s.data.num3 == 0 && s.data.num4 == 0 && s.data.num5 == 0 && s.data.num6 == 0) == responses.Count ? 0 : responses.Count(s => s.data.num1 == 100 && s.data.num2 == 0 && s.data.num3 == 0 && s.data.num4 == 0 && s.data.num5 == 0 && s.data.num6 == 0);
                            count++;
                            var dataList = new MachineSateCount
                            {
                                hour1 = Math.Round((responses.Sum(s => s.data.hour1)), 2),
                                hour2 = Math.Round((responses.Sum(s => s.data.hour2)), 2),
                                hour3 = Math.Round((responses.Sum(s => s.data.hour3)), 2),
                                hour4 = Math.Round((responses.Sum(s => s.data.hour4)), 2),
                                hour5 = Math.Round((responses.Sum(s => s.data.hour5)), 2),
                                hour6 = Math.Round((responses.Sum(s => s.data.hour6)), 2),
                                num1 = responses.Sum(s => s.data.hour1) == 0 ? 100 : Math.Round((responses.Sum(s => s.data.hour1)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                                num2 = responses.Sum(s => s.data.hour2) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour2)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                                num3 = responses.Sum(s => s.data.hour3) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour3)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                                num4 = responses.Sum(s => s.data.hour4) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour4)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                                num5 = responses.Sum(s => s.data.hour5) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour5)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                                num6 = responses.Sum(s => s.data.hour6) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour6)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                                datetime = responses.First().datetime
                            };

                            responsesList.Add(new MachineCanvasDataInfo() { data = dataList, datetime = responses.First().datetime });
                            responses.Clear();
                        }
                        else
                        {
                            start = dates[count];
                            var dts = start.AddMonths(1);
                            dts = dts.AddDays(0 - dts.Day).AddDays(1).AddSeconds(-1);
                            end = dts > dtEnd ? dtEnd.AddDays(1).AddSeconds(-1) : dts;//到月底
                            dateTime = end.ToString("yyyy-MM");
                            responsesList.Add(await SliceManager.GetMachineSlice(start, end, machinIds, CompanyId, ProjectId,
                            dateTime, 2));
                            count++;
                        }
                    }
                }
            }

            if (request.xtype == 1)
            {
                try
                {

                    var sections = sectionIds.Any()
                        ? _machineRepository.GetLevelSectionsBySectionIds(sectionIds, ProjectId)
                        : _machineRepository.GetNCLevelSections(request.sectionId, ProjectId);
                    var levels = sections.Where(q => q.levelid == patchId).ToList();
                    if (levels.Any())

                    {
                        foreach (var levelSection in levels)
                        {
                            var machineIds = _machineRepository.GetNCLevelSectionMachineIds(levelSection.id, ProjectId);
                            var dateTime = levelSection.title;
                            if (request.frequency != 0)
                            {
                                var timeList = new List<ResponseStartEnd>();

                                var result = _sigerProjectShiftRepository.Get(s => s.projectid == ProjectId && s.status == (int)RowState.Valid && s.id == request.frequency);
                                timeList.Add(new ResponseStartEnd
                                {
                                    id = result.id,
                                    title = result.title,
                                    start = result.start_time,
                                    end = result.end_time
                                });

                                var interal = DateTimeHelper.GetInteral(patchId);
                                var dates = DateTimeHelper.GetDateTimes(dtStart, dtEnd, 0);

                                var count = 0;
                                foreach (var date in dates)
                                {
                                    DateTime start;
                                    DateTime end;


                                    start = date;
                                    end = date.AddDays(interal).AddSeconds(-1) > DateTime.Now
                                        ? DateTime.Now
                                        : date.AddDays(interal).AddSeconds(-1);


                                    foreach (var classDate in timeList)
                                    {
                                        if (start <= end)
                                        {
                                            if (classDate.end > classDate.start)
                                            {
                                                var startTime = UnixTimeHelper.GetTimeByUnix(classDate.start);
                                                var endTime = UnixTimeHelper.GetTimeByUnix(classDate.end);

                                                DateTime.TryParse(start.ToString(ParameterConstant.DateFormat) + " " + startTime, out start);
                                                DateTime.TryParse(end.ToString(ParameterConstant.DateFormat) + " " + endTime, out end); ;

                                                timeDifference = (int)(end - start).TotalHours;
                                                responses.Add(await SliceManager.GetMachineSlice(start, end, machineIds, CompanyId, ProjectId,
                                                dateTime, 2));
                                            }
                                            else
                                            {
                                                var startTime = UnixTimeHelper.GetTimeByUnix(classDate.start);
                                                var endTime = UnixTimeHelper.GetTimeByUnix(classDate.end);

                                                DateTime.TryParse(start.AddDays(1).ToString(ParameterConstant.DateFormat) + " " + endTime, out end);
                                                DateTime.TryParse(start.ToString(ParameterConstant.DateFormat) + " " + startTime, out start);

                                                timeDifference = (int)(end - start).TotalHours;
                                                responses.Add(await SliceManager.GetMachineSlice(start, end, machineIds, CompanyId, ProjectId,
                                                dateTime, 2));
                                            }
                                        }
                                    }
                                }
                                var breakMathine = responses.Count(s => s.data.num1 == 100 && s.data.num2 == 0 && s.data.num3 == 0 && s.data.num4 == 0 && s.data.num5 == 0 && s.data.num6 == 0) == responses.Count ? 0 : responses.Count(s => s.data.num1 == 100 && s.data.num2 == 0 && s.data.num3 == 0 && s.data.num4 == 0 && s.data.num5 == 0 && s.data.num6 == 0);
                                count++;
                                var dataList = new MachineSateCount
                                {
                                    hour1 = Math.Round((responses.Sum(s => s.data.hour1)), 2),
                                    hour2 = Math.Round((responses.Sum(s => s.data.hour2)), 2),
                                    hour3 = Math.Round((responses.Sum(s => s.data.hour3)), 2),
                                    hour4 = Math.Round((responses.Sum(s => s.data.hour4)), 2),
                                    hour5 = Math.Round((responses.Sum(s => s.data.hour5)), 2),
                                    hour6 = Math.Round((responses.Sum(s => s.data.hour6)), 2),
                                    num1 = responses.Sum(s => s.data.hour1) == 0 ? 100 : Math.Round((responses.Sum(s => s.data.hour1)) / ((responses.Count - breakMathine) * timeDifference * machineIds.Count()) * 100, 2),
                                    num2 = responses.Sum(s => s.data.hour2) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour2)) / ((responses.Count - breakMathine) * timeDifference * machineIds.Count()) * 100, 2),
                                    num3 = responses.Sum(s => s.data.hour3) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour3)) / ((responses.Count - breakMathine) * timeDifference * machineIds.Count()) * 100, 2),
                                    num4 = responses.Sum(s => s.data.hour4) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour4)) / ((responses.Count - breakMathine) * timeDifference * machineIds.Count()) * 100, 2),
                                    num5 = responses.Sum(s => s.data.hour5) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour5)) / ((responses.Count - breakMathine) * timeDifference * machineIds.Count()) * 100, 2),
                                    num6 = responses.Sum(s => s.data.hour6) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour6)) / ((responses.Count - breakMathine) * timeDifference * machineIds.Count()) * 100, 2),
                                    datetime = responses.First().datetime
                                };

                                responsesList.Add(new MachineCanvasDataInfo() { data = dataList, datetime = responses.First().datetime });
                                responses.Clear();
                            }
                            else
                            {
                                responsesList.Add(await SliceManager.GetMachineSlice(dtStart, dtEnd, machineIds, CompanyId,
                                ProjectId, dateTime, 2));
                            }
                        }
                    }
                    else //最后一级
                    {
                        var machine = _machineRepository.GetNCMachineBySectionId(request.sectionId, ProjectId);
                        var dateTime = machine.name;
                        if (request.frequency != 0)
                        {
                            var timeList = new List<ResponseStartEnd>();

                            var result = _sigerProjectShiftRepository.Get(s => s.projectid == ProjectId && s.status == (int)RowState.Valid && s.id == request.frequency);
                            timeList.Add(new ResponseStartEnd
                            {
                                id = result.id,
                                title = result.title,
                                start = result.start_time,
                                end = result.end_time
                            });

                            var interal = DateTimeHelper.GetInteral(patchId);
                            var dates = DateTimeHelper.GetDateTimes(dtStart, dtEnd, 0);

                            var count = 0;
                            foreach (var date in dates)
                            {
                                DateTime start;
                                DateTime end;

                                start = date;
                                end = date.AddDays(interal).AddSeconds(-1) > DateTime.Now
                                    ? DateTime.Now
                                    : date.AddDays(interal).AddSeconds(-1);

                                if (start <= end)
                                {
                                    foreach (var classDate in timeList)
                                    {
                                        if (classDate.end > classDate.start)
                                        {
                                            var startTime = UnixTimeHelper.GetTimeByUnix(classDate.start);
                                            var endTime = UnixTimeHelper.GetTimeByUnix(classDate.end);

                                            DateTime.TryParse(start.ToString(ParameterConstant.DateFormat) + " " + startTime, out start);
                                            DateTime.TryParse(end.ToString(ParameterConstant.DateFormat) + " " + endTime, out end); ;
                                            timeDifference = (int)(end - start).TotalHours;

                                            responses.Add(await SliceManager.GetMachineSlice(start, end, new List<int> { machine.id }, CompanyId, ProjectId,
                                            dateTime, 2));
                                        }
                                        else
                                        {
                                            var startTime = UnixTimeHelper.GetTimeByUnix(classDate.start);
                                            var endTime = UnixTimeHelper.GetTimeByUnix(classDate.end);

                                            DateTime.TryParse(start.AddDays(1).ToString(ParameterConstant.DateFormat) + " " + endTime, out end);
                                            DateTime.TryParse(start.ToString(ParameterConstant.DateFormat) + " " + startTime, out start);
                                            timeDifference = (int)(end - start).TotalHours;

                                            responses.Add(await SliceManager.GetMachineSlice(start, end, new List<int> { machine.id }, CompanyId, ProjectId,
                                            dateTime, 2));
                                        }
                                    }
                                }
                            }
                            var breakMathine = responses.Count(s => s.data.num1 == 100 && s.data.num2 == 0 && s.data.num3 == 0 && s.data.num4 == 0 && s.data.num5 == 0 && s.data.num6 == 0 && s.data.hour1 == 0
                            && s.data.hour2 == 0 && s.data.hour3 == 0 && s.data.hour4 == 0 && s.data.hour5 == 0 && s.data.hour6 == 0) == responses.Count ? 0
                            : responses.Count(s => s.data.num1 == 100 && s.data.num2 == 0 && s.data.num3 == 0 && s.data.num4 == 0 && s.data.num5 == 0 && s.data.num6 == 0 && s.data.hour1 == 0
                            && s.data.hour2 == 0 && s.data.hour3 == 0 && s.data.hour4 == 0 && s.data.hour5 == 0 && s.data.hour6 == 0);

                            count++;
                            var dataList = new MachineSateCount
                            {
                                hour1 = Math.Round((responses.Sum(s => s.data.hour1)), 2),
                                hour2 = Math.Round((responses.Sum(s => s.data.hour2)), 2),
                                hour3 = Math.Round((responses.Sum(s => s.data.hour3)), 2),
                                hour4 = Math.Round((responses.Sum(s => s.data.hour4)), 2),
                                hour5 = Math.Round((responses.Sum(s => s.data.hour5)), 2),
                                hour6 = Math.Round((responses.Sum(s => s.data.hour6)), 2),
                                num1 = responses.Sum(s => s.data.hour1) == 0 ? 100 : Math.Round((responses.Sum(s => s.data.hour1)) / ((responses.Count - breakMathine) * timeDifference) * 100, 2),
                                num2 = responses.Sum(s => s.data.hour2) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour2)) / ((responses.Count - breakMathine) * timeDifference) * 100, 2),
                                num3 = responses.Sum(s => s.data.hour3) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour3)) / ((responses.Count - breakMathine) * timeDifference) * 100, 2),
                                num4 = responses.Sum(s => s.data.hour4) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour4)) / ((responses.Count - breakMathine) * timeDifference) * 100, 2),
                                num5 = responses.Sum(s => s.data.hour5) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour5)) / ((responses.Count - breakMathine) * timeDifference) * 100, 2),
                                num6 = responses.Sum(s => s.data.hour6) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour6)) / ((responses.Count - breakMathine) * timeDifference) * 100, 2),
                                datetime = responses.First().datetime
                            };
                            responsesList.Add(new MachineCanvasDataInfo() { data = dataList, datetime = responses.First().datetime });
                            responses.Clear();
                        }
                        else
                        {
                            responsesList.Add(await SliceManager.GetMachineSlice(dtStart, dtEnd, new List<int> { machine.id },
                            CompanyId, ProjectId, dateTime, 2));
                        }
                    }
                }
                catch
                {
                    throw new BadRequestException(CommonEnum.RecordNotFound);
                }
            }
            if (request.xtype == 2)
            {
                if (request.frequency == 0)
                {
                    var data = _configFrequencyRepository.Get(s => s.projectid == ProjectId && s.status == (int)RowState.Valid);
                    var train = data.train.Split(new char[] { ',' });
                    var timeList = new List<ResponseStartEnd>();
                    var index = 1;
                    foreach (var item in train)
                    {
                        var result = _sigerProjectShiftRepository.Get(s => s.projectid == ProjectId && s.status == (int)RowState.Valid && s.id.ToString() == item);
                        timeList.Add(new ResponseStartEnd
                        {
                            title = result.title,
                            id = index,
                            start = result.start_time,
                            end = result.end_time
                        });
                        index++;
                    }

                    var interal = DateTimeHelper.GetInteral(patchId);
                    var dates = DateTimeHelper.GetDateTimes(dtStart, dtEnd, 0);

                    var count = 0;
                    foreach (var classDate in timeList)
                    {
                        foreach (var date in dates)
                        {
                            DateTime start;
                            DateTime end;
                            string dateTime;

                            start = date;
                            end = date.AddDays(interal).AddSeconds(-1) > DateTime.Now
                                ? DateTime.Now
                                : date.AddDays(interal).AddSeconds(-1);

                            timeList = timeList.OrderByDescending(x => x.id).ToList();

                            dateTime = classDate.title;
                            if (start <= end)
                            {
                                if (classDate.end > classDate.start)
                                {
                                    var startTime = UnixTimeHelper.GetTimeByUnix(classDate.start);
                                    var endTime = UnixTimeHelper.GetTimeByUnix(classDate.end);

                                    DateTime.TryParse(start.ToString(ParameterConstant.DateFormat) + " " + startTime, out DateTime startClass);
                                    DateTime.TryParse(end.ToString(ParameterConstant.DateFormat) + " " + endTime, out DateTime endClass); ;
                                    timeDifference = (int)(endClass - startClass).TotalHours;
                                    responses.Add(await SliceManager.GetMachineSlice(startClass, endClass, machinIds, CompanyId, ProjectId,
                                    dateTime, 2));
                                    //count++;
                                }
                                else
                                {
                                    var startTime = UnixTimeHelper.GetTimeByUnix(classDate.start);
                                    var endTime = UnixTimeHelper.GetTimeByUnix(classDate.end);

                                    DateTime.TryParse(start.AddDays(1).ToString(ParameterConstant.DateFormat) + " " + endTime, out DateTime endClass);
                                    DateTime.TryParse(start.ToString(ParameterConstant.DateFormat) + " " + startTime, out DateTime startClass);
                                    timeDifference = (int)(endClass - startClass).TotalHours;
                                    responses.Add(await SliceManager.GetMachineSlice(startClass, endClass, machinIds, CompanyId, ProjectId,
                                    dateTime, 2));
                                    //count++;
                                }
                            }
                        }
                        var breakMathine = responses.Count(s => s.data.num1 == 100 && s.data.num2 == 0 && s.data.num3 == 0 && s.data.num4 == 0 && s.data.num5 == 0 && s.data.num6 == 0) == responses.Count ? 0 : responses.Count(s => s.data.num1 == 100 && s.data.num2 == 0 && s.data.num3 == 0 && s.data.num4 == 0 && s.data.num5 == 0 && s.data.num6 == 0);
                        count++;
                        var dataList = new MachineSateCount
                        {
                            hour1 = Math.Round((responses.Sum(s => s.data.hour1)), 2),
                            hour2 = Math.Round((responses.Sum(s => s.data.hour2)), 2),
                            hour3 = Math.Round((responses.Sum(s => s.data.hour3)), 2),
                            hour4 = Math.Round((responses.Sum(s => s.data.hour4)), 2),
                            hour5 = Math.Round((responses.Sum(s => s.data.hour5)), 2),
                            hour6 = Math.Round((responses.Sum(s => s.data.hour6)), 2),
                            num1 = responses.Sum(s => s.data.hour1) == 0 ? 100 : Math.Round((responses.Sum(s => s.data.hour1)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                            num2 = responses.Sum(s => s.data.hour2) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour2)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                            num3 = responses.Sum(s => s.data.hour3) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour3)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                            num4 = responses.Sum(s => s.data.hour4) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour4)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                            num5 = responses.Sum(s => s.data.hour5) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour5)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                            num6 = responses.Sum(s => s.data.hour6) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour6)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                            datetime = responses.First().datetime
                        };

                        responsesList.Add(new MachineCanvasDataInfo() { data = dataList, datetime = responses.First().datetime });
                        responses.Clear();

                    }
                }
                else
                {
                    var timeList = new List<ResponseStartEnd>();

                    var result = _sigerProjectShiftRepository.Get(s => s.projectid == ProjectId && s.status == (int)RowState.Valid && s.id == request.frequency);
                    timeList.Add(new ResponseStartEnd
                    {
                        id = result.id,
                        title = result.title,
                        start = result.start_time,
                        end = result.end_time
                    });

                    var interal = DateTimeHelper.GetInteral(patchId);
                    var dates = DateTimeHelper.GetDateTimes(dtStart, dtEnd, 0);

                    var count = 0;
                    foreach (var classDate in timeList)
                    {
                        foreach (var date in dates)
                        {
                            DateTime start;
                            DateTime end;
                            string dateTime;

                            start = date;
                            end = date.AddDays(interal).AddSeconds(-1) > DateTime.Now
                                ? DateTime.Now
                                : date.AddDays(interal).AddSeconds(-1);


                            dateTime = classDate.title;
                            if (start <= end)
                            {
                                if (classDate.end > classDate.start)
                                {
                                    var startTime = UnixTimeHelper.GetTimeByUnix(classDate.start);
                                    var endTime = UnixTimeHelper.GetTimeByUnix(classDate.end);

                                    DateTime.TryParse(start.ToString(ParameterConstant.DateFormat) + " " + startTime, out DateTime startClass);
                                    DateTime.TryParse(end.ToString(ParameterConstant.DateFormat) + " " + endTime, out DateTime endClass); ;
                                    timeDifference = (int)(endClass - startClass).TotalHours;

                                    responses.Add(await SliceManager.GetMachineSlice(startClass, endClass, machinIds, CompanyId, ProjectId,
                                    dateTime, 2));
                                    //count++;
                                }
                                else
                                {
                                    var startTime = UnixTimeHelper.GetTimeByUnix(classDate.start);
                                    var endTime = UnixTimeHelper.GetTimeByUnix(classDate.end);

                                    DateTime.TryParse(start.AddDays(1).ToString(ParameterConstant.DateFormat) + " " + endTime, out DateTime endClass);
                                    DateTime.TryParse(start.ToString(ParameterConstant.DateFormat) + " " + startTime, out DateTime startClass);
                                    timeDifference = (int)(endClass - startClass).TotalHours;

                                    responses.Add(await SliceManager.GetMachineSlice(startClass, endClass, machinIds, CompanyId, ProjectId,
                                    dateTime, 2));
                                    //count++;
                                }
                            }
                        }
                        var breakMathine = responses.Count(s => s.data.num1 == 100 && s.data.num2 == 0 && s.data.num3 == 0 && s.data.num4 == 0 && s.data.num5 == 0 && s.data.num6 == 0) == responses.Count ? 0 : responses.Count(s => s.data.num1 == 100 && s.data.num2 == 0 && s.data.num3 == 0 && s.data.num4 == 0 && s.data.num5 == 0 && s.data.num6 == 0);
                        count++;
                        var dataList = new MachineSateCount
                        {
                            hour1 = Math.Round((responses.Sum(s => s.data.hour1)), 2),
                            hour2 = Math.Round((responses.Sum(s => s.data.hour2)), 2),
                            hour3 = Math.Round((responses.Sum(s => s.data.hour3)), 2),
                            hour4 = Math.Round((responses.Sum(s => s.data.hour4)), 2),
                            hour5 = Math.Round((responses.Sum(s => s.data.hour5)), 2),
                            hour6 = Math.Round((responses.Sum(s => s.data.hour6)), 2),
                            num1 = responses.Sum(s => s.data.hour1) == 0 ? 100 : Math.Round((responses.Sum(s => s.data.hour1)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                            num2 = responses.Sum(s => s.data.hour2) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour2)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                            num3 = responses.Sum(s => s.data.hour3) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour3)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                            num4 = responses.Sum(s => s.data.hour4) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour4)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                            num5 = responses.Sum(s => s.data.hour5) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour5)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                            num6 = responses.Sum(s => s.data.hour6) == 0 ? 0 : Math.Round((responses.Sum(s => s.data.hour6)) / ((responses.Count - breakMathine) * timeDifference * machinIds.Count) * 100, 2),
                            datetime = responses.First().datetime
                        };

                        responsesList.Add(new MachineCanvasDataInfo() { data = dataList, datetime = responses.First().datetime });
                        responses.Clear();
                    }
                }
            }

            if (request.toexcel == 0)
            {
                return new ObjectResult(responsesList);
            }

            return ExportBarChart(responsesList);
        }

        /// <summary>
        /// 查询班次接口
        /// </summary>
        /// <returns></returns>
        [HttpGet]
        public IActionResult GetClass()
        {
            var timeList = new List<ResponseStartEnd>();
            var data = _configFrequencyRepository.Get(s => s.projectid == ProjectId && s.status == (int)RowState.Valid);
            if (data!=null)
            {
                var train = data.train.Split(new char[] { ',' });
                foreach (var item in train)
                {
                    var result = _sigerProjectShiftRepository.Get(s => s.projectid == ProjectId && s.status == (int)RowState.Valid && s.id.ToString() == item);
                    timeList.Add(new ResponseStartEnd
                    {
                        id = result.id,
                        start = result.start_time,
                        end = result.end_time,
                        title = result.title
                    });
                }
            }
            return new ObjectResult(timeList);
        }

        /// <summary>
        /// 设备切片分析
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<IActionResult> MachineCanvasDataInfoCJ([FromBody] RequestGetMachineCanvasDataInfo request)
        {
            //xtype - 0:时间 1：层级
            //x_ID - 0:天 1：周 2：月 xtype=1时，为levelid
            if (!DateTime.TryParse(request.startTime, out var dtStart))
            {
                dtStart = DateTime.Now.AddDays(-14);
            }

            if (!DateTime.TryParse(request.endTime, out var dtEnd))
            {
                dtEnd = DateTime.Now;
            }

            if (dtStart > dtEnd)
            {
                throw new ServerException(100158);
            }

            if (dtStart == dtEnd)
            {
                dtStart = DateTime.Parse(dtStart.ToString(ParameterConstant.DateFromZero));
                dtEnd = DateTime.Parse(dtEnd.ToString(ParameterConstant.DateToEnd));
            }

            dtEnd = dtEnd > DateTime.Now ? DateTime.Now : dtEnd;

            var machinIds = new List<int>();
            var sectionIds = string.IsNullOrWhiteSpace(request.sectionIdStr)
                ? new List<int>()
                : request.sectionIdStr.Split(',', StringSplitOptions.RemoveEmptyEntries).Select(t => t.ToInt())
                    .ToList();
            machinIds = sectionIds.Any()
                ? _machineRepository.GetNCMachineIdsBySectionIds(sectionIds, ProjectId).ToList()
                : _machineRepository.GetNCLevelSectionMachineIds(request.sectionId, ProjectId).ToList();

            var responses = new List<MachineCanvasDataInfo>();
            var patchId = request.x_id.ToInt();
            if (request.xtype == 0)
            {
                var type = 0;
                if (patchId == 1)
                {
                    type = 11;
                }
                else if (patchId == 2)
                {
                    type = 12;
                }
                else
                {
                    type = patchId;
                }
                var interal = DateTimeHelper.GetInteral(patchId);
                var dates = DateTimeHelper.GetDateTimes(dtStart, dtEnd, type);
                var xData = DateTimeHelper.GetStrDateTimes(dtStart, dtEnd, type);
                if (dates.Count() > 7)
                {
                    dates.RemoveAt(0);
                    xData.RemoveAt(0);
                }
                var count = 0;
                foreach (var date in dates)
                {
                    DateTime start;
                    DateTime end;
                    string dateTime;
                    if (patchId == 0)
                    {
                        start = date;
                        end = date.AddDays(interal).AddSeconds(-1) > DateTime.Now
                            ? DateTime.Now
                            : date.AddDays(interal).AddSeconds(-1);
                        dateTime = end.ToString(ParameterConstant.DateFormat);
                    }
                    else if (patchId == 1)
                    {
                        //start = date;
                        //end = date.AddDays(interal).AddSeconds(-1) > DateTime.Now
                        //    ? DateTime.Now
                        //    : date.AddDays(interal).AddSeconds(-1);
                        dateTime = xData[count];
                        var weekStart = DateTimeHelper.GetWeekRangeStart(dateTime);
                        var weekEnd = DateTimeHelper.GetWeekRangeEnd(dateTime);
                        start = dtStart > weekStart ? dtStart : weekStart;
                        end = dtEnd < weekEnd ? dtEnd : weekEnd.AddDays(1).AddSeconds(-1);
                    }
                    else
                    {
                        start = dates[count];
                        var dts = start.AddMonths(1);
                        dts = dts.AddDays(0 - dts.Day).AddDays(1).AddSeconds(-1);
                        end = dts > dtEnd ? dtEnd.AddDays(1).AddSeconds(-1) : dts;//到月底
                        dateTime = end.ToString("yyyy-MM");
                    }

                    responses.Add(await SliceManager.GetMachineSlice(start, end, machinIds, CompanyId, ProjectId,
                        dateTime, 2));
                    count++;
                }
            }

            if (request.xtype == 1)
            {
                try
                {

                    var sections = sectionIds.Any()
                        ? _machineRepository.GetLevelSectionsBySectionIds(sectionIds, ProjectId)
                        : _machineRepository.GetNCLevelSections(request.sectionId, ProjectId);
                    var levels = sections.Where(q => q.levelid == patchId).ToList();
                    if (levels.Any())
                    {
                        foreach (var levelSection in levels)
                        {
                            var machineIds = _machineRepository.GetNCLevelSectionMachineIds(levelSection.id, ProjectId);
                            var dateTime = levelSection.title;
                            responses.Add(await SliceManager.GetMachineSlice(dtStart, dtEnd, machineIds, CompanyId,
                                ProjectId, dateTime, 2));
                        }
                    }
                    else //最后一级
                    {
                        var machine = _machineRepository.GetNCMachineBySectionId(request.sectionId, ProjectId);
                        var dateTime = machine.name;
                        responses.Add(await SliceManager.GetMachineSlice(dtStart, dtEnd, new List<int> { machine.id },
                            CompanyId, ProjectId, dateTime, 2));
                    }
                }
                catch
                {
                    throw new BadRequestException(CommonEnum.RecordNotFound);
                }
            }

            if (request.toexcel == 0)
            {
                return new ObjectResult(responses);
            }

            return ExportBarChart(responses);
        }

        [HttpPost]
        public IActionResult ExportBarChart(List<MachineCanvasDataInfo> responses)
        {
            if (!responses.Any())
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }

           
            var rootDir = FileSystemHelper.GetPhysicalFolders(FileSystemHelper.CommonFileSetting.PhysicalFolder, FileSystemHelper.ExportFileName);

            var temporaryFileName = $"MachineState_{DateTime.Now:yyyyMMddHHmmss}.xlsx";
            var fileName = Path.Combine(rootDir, temporaryFileName);

            var helper =
                new EpPlusExcelHelper<MachineStateEchartList>(_machineRepository.GetProjectLanguage(ProjectId));
            try
            {
                var machineYields = new List<MachineStateEchartList>();
                var index = 1;
                foreach (var yield in responses)
                {
                    var machineYield = new MachineStateEchartList
                    {
                        Abscissa = yield.datetime,
                        Shutdown = yield.data.num1,
                        Run = yield.data.num2,
                        Debug = yield.data.num3,
                        Idle = yield.data.num4,
                        Fault = yield.data.num5,
                    };
                    machineYields.Add(machineYield);
                    index++;
                }

                helper.GenerateExcel(machineYields, fileName, false, EchartCategory.SlicePercentColumnStacked);
                return new ObjectResult($"{FileSystemHelper.CommonFileSetting.RequestPath}/{FileSystemHelper.ExportFileName}/{temporaryFileName}");
            }
            catch (Exception e)
            {
                Logger.WriteLineError("Export MachineState failed, error: " + e.Message);
                throw new BadRequestException(RequestEnum.ExportFailed);
            }
            finally
            {
                helper.Dispose();
            }
        }

        /// <summary>
        /// 状态管理列表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<IActionResult> GetMachineSateList([FromBody] RequestGetAbnormalList request)
        {
            Logger.WriteLineError("section time, error1: " + DateTime.Now.ToString());
            if (request.toexcel != 0)
            {
                request.pagesize = 0;
            }
            var condition = new MachineSateCondition();
            if (request.sectionID != 0)
            {
                condition.MachinIds = _machineRepository.GetNCLevelSectionMachineIds(request.sectionID, ProjectId);
            }
            Logger.WriteLineError("section time, error2: " + DateTime.Now.ToString());
            condition.BigSmall = request.big_small.ToInt();
            condition.BigSmallSpan = request.big_small_span;
            condition.StartTime = request.startTime;
            condition.EndTime = request.endTime;
            condition.ErrType = string.IsNullOrWhiteSpace(request.errType) ? -1 : request.errType.ToInt();
            condition.Page = request.page;
            condition.PageSize = request.pagesize;

            var pageInfo = new Pagination(request.page, request.pagesize);
            var repositoey = new SliceSateRepository(CompanyId, ProjectId);
            var sates = repositoey.GetCncSliceSatesForEditAsync(pageInfo, condition);
            var result = new List<ResponseGetMachineSateList>();
            //合并行
            if (!sates.Value.Any())
            {
                return new PagedObjectResult(result, 0, request.page, request.pagesize);
            }
            Logger.WriteLineError("section time, error3: " + DateTime.Now.ToString());
            var stations = _machineRepository.GetNCMachinSectionInfos(condition.MachinIds, ProjectId).ToList();
            Logger.WriteLineError("section time, error4: " + DateTime.Now.ToString());
            foreach (var sate in sates.Value)
            {
                var station = stations.FirstOrDefault(q => q.machineId == sate.MachineID);
                var entity = new ResponseGetMachineSateList
                {
                    end_time = sate.EndTime.ToString(ParameterConstant.DateTimeFormat),
                    start_time = sate.StartTime.ToString(ParameterConstant.DateTimeFormat),
                    machine_status = sate.Status,
                    machineID = sate.MachineID,
                    id = sate.id,
                    time_span = (sate.EndTime - sate.StartTime).TotalSeconds,
                    sectionID = station == null ? "" : station.section_name + "-" + station.station_name,
                    info = "",
                    remarks = string.IsNullOrWhiteSpace(sate.Remark) ? "" : sate.Remark,
                    user = "",
                    userId = 0
                };
                if (!string.IsNullOrWhiteSpace(sate.UserName))
                {
                    var user = _userRepository.Get(q =>
                        q.mid == sate.UserName.ToInt() && q.projectid == ProjectId && q.status == (int)RowState.Valid);
                    if (user != null)
                    {
                        entity.userId = user.mid;
                        entity.user = user.name;
                    }
                }

                if (sate.WordingModeId != 0)
                {
                    var mode = _modeRepository.Get(sate.WordingModeId);
                    if (mode != null && mode.status == (int)RowState.Valid)
                    {
                        entity.workModeId = mode.id;
                        entity.info = mode.workingmodename;
                        var parentMode = _wordingModeRepository.Get(t =>
                            t.id == mode.parentid && t.projectid == ProjectId && t.status == (int)RowState.Valid);
                        if (parentMode != null)
                        {
                            entity.workModeParentId = parentMode.id;
                            entity.workModeParnetName = parentMode.workingmodename;
                        }
                    }
                }

                result.Add(entity);
            }
            Logger.WriteLineError("section time, error5: " + DateTime.Now.ToString());
            if (request.toexcel == 0)
            {
                return new PagedObjectResult(result, sates.Key.TotalItemCount, request.page, request.pagesize);
            }
            return await Task.Run(() =>
            {
                return ExeclMachineSateList(result);
            });
        }

        [HttpPost]
        public IActionResult ExeclMachineSateList(List<ResponseGetMachineSateList> result)
        {
            var helper = new EpPlusExcelHelper<ExeclMachineSateList>();
            try
            {
              
                var rootDir = FileSystemHelper.GetPhysicalFolders(FileSystemHelper.CommonFileSetting.PhysicalFolder, FileSystemHelper.ExportFileName);
                var temporaryFileName = $"设备状态管理_{DateTime.Now:yyyyMMddHHmmss}.xlsx";
                var fileName = Path.Combine(rootDir, temporaryFileName);
                var datalist = new List<ExeclMachineSateList>();
                int index = 1;
                foreach (var item in result)
                {
                    var model = Mapper<ResponseGetMachineSateList, ExeclMachineSateList>.Map(item);
                    model.No = index;
                    model.time_span = GetTimeBySecond(Convert.ToInt32(item.time_span));
                    switch (item.machine_status)
                    {
                        case 0:
                            model.machine_status = "关机";
                            break;
                        case 1:
                            model.machine_status = "运行";
                            break;
                        case 2:
                            model.machine_status = "手动/调试";
                            break;
                        case 3:
                            model.machine_status = "空闲";
                            break;
                        case 4:
                            model.machine_status = "报警";
                            break;
                    }
                    datalist.Add(model);
                    index++;
                }

                helper.GenerateExcel(datalist, fileName);
                return new ObjectResult($"{FileSystemHelper.CommonFileSetting.RequestPath}/{FileSystemHelper.ExportFileName}/{temporaryFileName}");
            }
            catch (Exception e)
            {
                Logger.WriteLineError("ExportMachineType failed, error: " + e.Message);
                throw new BadRequestException(RequestEnum.ExportFailed);
            }
            finally
            {
                helper.Dispose();
            }
        }
        public static string GetTimeBySecond(int totalSecond)
        {
            var timeStr = "";
            if (totalSecond == 0)
            {
                timeStr = "0";
            }

            var day = totalSecond / 86400;
            if (day > 0)
            {
                timeStr += day + "天";
            }

            var hour = (totalSecond % 86400) / 3600;
            if (hour > 0)
            {
                timeStr += hour + "小时";
            }

            var min = (((totalSecond % 86400) % 3600) / 60);
            if (min > 0)
            {
                timeStr += min + "分钟";
            }

            var sec = ((totalSecond % 86400) % 3600) % 60;
            if (sec > 0)
            {
                timeStr += sec + "秒";
            }

            return timeStr;
        }
        private static async Task<List<CncSliceSate>> GroupedSates(IEnumerable<CncSliceSate> sates)
        {
            return await Task.Run(() =>
            {
                var tempSates = new List<CncSliceSate>();
                if (sates.Count() > 1)
                {
                    var sateList = sates.ToList();
                    var groupBy = 0;
                    for (var i = 0; i < sateList.Count - 1; i++)
                    {
                        var temp = sateList[i];
                        var tempNext = sateList[i + 1];
                        temp.GroupBy = groupBy;
                        if (temp.MachineID != tempNext.MachineID || temp.EndTime != tempNext.StartTime)
                        {
                            groupBy++;
                        }

                        tempSates.Add(temp);
                    }

                    var lastSateList = sateList.Last();
                    var tempLast = tempSates.Last();
                    if (lastSateList.MachineID != tempLast.MachineID || lastSateList.StartTime != tempLast.EndTime)
                    {
                        lastSateList.GroupBy = groupBy + 1;
                    }
                    else
                    {
                        lastSateList.GroupBy = groupBy;
                    }

                    tempSates.Add(lastSateList);
                }
                else if (sates.Count() == 1)
                {
                    tempSates.Add(sates.First());
                }

                var groups = from p in tempSates
                             group p by new
                             {
                                 p.GroupBy
                             }
                    into g
                             select g.ToList();

                var temps = new List<CncSliceSate>();
                foreach (var groupedSate in groups)
                {
                    var list = groupedSate.ToList();
                    var first = list[0];
                    if (list.Count == 1)
                    {
                        temps.Add(first);
                    }
                    else
                    {
                        var temp = new CncSliceSate
                        {
                            StartTime = first.StartTime,
                            EndTime = list[list.Count - 1].EndTime,
                            MachineID = first.MachineID,
                            Status = first.Status,
                            id = first.id,
                            Remark = first.Remark,
                            UserName = first.UserName,
                            WordingModeId = first.WordingModeId
                        };
                        temps.Add(temp);
                    }
                }

                return temps;
            });
        }

        [HttpPost]
        public async Task<IActionResult> AppendMode([FromBody] RequestAppendMode request)
        {
            var repositoey = new SliceSateRepository(CompanyId, ProjectId);
            var result = await repositoey.UpdateRemarkAsync(request.id.ToInt(), request.edit_Working_mode_id.ToInt(),
                request.edit_personnel_id,
                request.edit_node, request.startTime, request.endTime);

            if (result <= 0)
            {
                throw new BadRequestException(CommonEnum.Fail);
            }

            return new ObjectResult(CommonEnum.Succefull);
        }

        [HttpGet]
        public IActionResult GetMachinesByStationId(int stationId)
        {
            var machines = _machineRepository.GetNCMachinStationInfos(ProjectId);
            return new ObjectResult(machines.Where(m => m.station_Id == stationId).ToList());
        }

        /// <summary>
        /// 状态异常统计
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<IActionResult> MachineStatusErrorSummary([FromBody] RequestMachineErrorSummary request)
        {
            //errorType:  2手动调试 3空闲 4报警 
            //x_id 1时间范围 2产线 3录入人员 4工作模式
            //levelId :x_id = 1时： 1日 2周 3月 4年；x_Id = 2时：产线级别
            var dateTimes = request.datetime.Split(" - ");
            if (dateTimes.Length != 2)
            {
                throw new BadRequestException(CncEnum.DateTimeFormatError);
            }

            var success = DateTime.TryParse(dateTimes[0], out var dtStart);
            if (!success)
            {
                throw new BadRequestException(CncEnum.DateTimeFormatError);
            }

            success = DateTime.TryParse(dateTimes[1], out var dtEnd);
            if (!success)
            {
                throw new BadRequestException(CncEnum.DateTimeFormatError);
            }

            dtEnd = dtEnd.AddDays(1).AddSeconds(-1);
            var result = new List<ResponseMachineErrorSummary>();

            var xid = request.x_id.ToInt(); // 横坐标
            var levelId = request.levelId.ToInt();
            var errorType = request.errorType.ToInt();

            var machineIds = _machineRepository.GetNCMachinIdsBySectionId(request.sectionId, 0, ProjectId);

            var statusSettings = _settingRepository.GetMachineSettings(machineIds, ProjectId);

            var cncExceptionRepository = new SliceSateRepository(CompanyId, ProjectId);
            var exceptions = await cncExceptionRepository.GetCncSliceSatesForAnalysisAsync(machineIds, errorType,
                dtStart.ToString(ParameterConstant.DateTimeFormat),
                dtEnd.ToString(ParameterConstant.DateTimeFormat));
            if (exceptions.Any())
            {
                if (xid == 1) //时间范围
                {
                    if (levelId == 1)
                    {
                        result.AddRange(AddUpTimeByTime(dtStart, dtEnd, 1, errorType, exceptions, statusSettings));
                    }

                    if (levelId == 2)
                    {
                        result.AddRange(AddUpTimeByTime(dtStart, dtEnd, 7, errorType, exceptions, statusSettings));
                    }

                    if (levelId == 3)
                    {
                        result.AddRange(AddUpTimeByTime(dtStart, dtEnd, 30, errorType, exceptions, statusSettings));
                    }
                }
                else if (xid == 2) // 产线
                {
                    //取sectionID下面的设备
                    var sections = _machineRepository.GetNCLevelSections(request.sectionId, ProjectId);
                    var levels = sections.Where(q => q.levelid == levelId).ToList();

                    foreach (var levelSection in levels)
                    {
                        var machines = _machineRepository.GetNCLevelSectionMachineIds(levelSection.id, ProjectId);
                        var query = exceptions.Where(q => machines.Contains(q.MachineID));

                        var response = AddUpTimeByType(errorType, levelSection.title, query, dtStart, dtEnd,
                            statusSettings);

                        result.Add(response);
                    }
                }
                else if (xid == 3) //人员
                {
                    var userExceptions = exceptions.Where(q => !string.IsNullOrEmpty(q.UserName));
                    Logger.WriteLineError("Test  : " + userExceptions.ToList().Count);
                    var groups = from p in userExceptions
                                 group p by new
                                 {
                                     p.UserName
                                 }
                        into g
                                 select g;

                    foreach (var error in groups)
                    {
                        Logger.WriteLineError("Test1  : " + error.ToStr());
                        var user = _userRepository.Get(q => q.mid == error.Key.UserName.ToInt());
                        if (user != null && user.status == (int)RowState.Valid)
                        {
                            var type = AddUpTimeByType(errorType, user.name, error.ToList(), dtStart, dtEnd,
                                statusSettings);
                            Logger.WriteLineError("Test2  : " + type.ToStr());
                            result.Add(AddUpTimeByType(errorType, user.name, error.ToList(), dtStart, dtEnd,
                                statusSettings));
                        }
                    }
                }
                else if (xid == 4) //工作模式
                {
                    var groups = from p in exceptions.Where(q => q.WordingModeId != 0)
                                 group p by new
                                 {
                                     p.WordingModeId
                                 }
                        into g
                                 select g;
                    foreach (var error in groups)
                    {
                        var mode = _wordingModeRepository.Get(error.Key.WordingModeId);
                        if (mode != null && mode.status == (int)RowState.Valid)
                        {
                            result.Add(AddUpTimeByType(errorType, mode.workingmodename, error.ToList(), dtStart, dtEnd,
                                statusSettings));
                        }
                    }
                }
            }

            if (request.toexcel == 0)
            {
                return new ObjectResult(result);
            }

            switch (xid)
            {
                case 1:
                    return ExportAlarmDataForAnalysis(result, (int)XType.Date);
                case 2:
                    return ExportAlarmDataForAnalysis(result, (int)XType.section);
                case 3:
                    return ExportAlarmDataForAnalysis(result, (int)XType.User);
                case 4:
                    return ExportAlarmDataForAnalysis(result, (int)XType.WorkMode);
                default:
                    return ExportAlarmDataForAnalysis(result, (int)XType.Other);
            }
        }

        [HttpPost]
        private IActionResult ExportAlarmDataForAnalysis(List<ResponseMachineErrorSummary> responses, int type)
        {
            if (!responses.Any())
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }


            var rootDir = FileSystemHelper.GetPhysicalFolders(FileSystemHelper.CommonFileSetting.PhysicalFolder, FileSystemHelper.ExportFileName);

            var temporaryFileName = $"machineyields_{DateTime.Now:yyyyMMddHHmmss}.xlsx";
            var fileName = Path.Combine(rootDir, temporaryFileName);

            dynamic helper;
            switch (type)
            {
                case 1:
                    helper = new EpPlusExcelHelper<AlarmDataForAnalysis1>(_machineRepository.GetProjectLanguage(ProjectId));
                    try
                    {
                        var machineYields = new List<AlarmDataForAnalysis1>();
                        var index = 1;
                        foreach (var yield in responses)
                        {
                            var machineYield = new AlarmDataForAnalysis1
                            {
                                No = index,
                                Abscissa = yield.title,
                                Total = yield.total,
                                Idle = yield.free,
                                Debug = yield.dubugging,
                                Fault = yield.fault,
                            };
                            machineYields.Add(machineYield);
                            index++;
                        }

                        helper.GenerateExcel(machineYields, fileName);
                        return new ObjectResult($"{FileSystemHelper.CommonFileSetting.RequestPath}/{FileSystemHelper.ExportFileName}/{temporaryFileName}");
                    }
                    catch (Exception e)
                    {
                        Logger.WriteLineError("Export AlarmDataForAnalysis yields failed, error: " + e.Message);
                        throw new BadRequestException(RequestEnum.ExportFailed);
                    }
                    finally
                    {
                        helper.Dispose();
                    }
                case 2:
                    helper = new EpPlusExcelHelper<AlarmDataForAnalysis2>(_machineRepository.GetProjectLanguage(ProjectId));
                    try
                    {
                        var machineYields = new List<AlarmDataForAnalysis2>();
                        var index = 1;
                        foreach (var yield in responses)
                        {
                            var machineYield = new AlarmDataForAnalysis2
                            {
                                No = index,
                                Abscissa = yield.title,
                                Total = yield.total,
                                Idle = yield.free,
                                Debug = yield.dubugging,
                                Fault = yield.fault,
                            };
                            machineYields.Add(machineYield);
                            index++;
                        }

                        helper.GenerateExcel(machineYields, fileName);
                        return new ObjectResult($"{FileSystemHelper.CommonFileSetting.RequestPath}/{FileSystemHelper.ExportFileName}/{temporaryFileName}");
                    }
                    catch (Exception e)
                    {
                        Logger.WriteLineError("Export AlarmDataForAnalysis yields failed, error: " + e.Message);
                        throw new BadRequestException(RequestEnum.ExportFailed);
                    }
                    finally
                    {
                        helper.Dispose();
                    }
                case 3:
                    helper = new EpPlusExcelHelper<AlarmDataForAnalysis3>(_machineRepository.GetProjectLanguage(ProjectId));
                    try
                    {
                        var machineYields = new List<AlarmDataForAnalysis3>();
                        var index = 1;
                        foreach (var yield in responses)
                        {
                            var machineYield = new AlarmDataForAnalysis3
                            {
                                No = index,
                                Abscissa = yield.title,
                                Total = yield.total,
                                Idle = yield.free,
                                Debug = yield.dubugging,
                                Fault = yield.fault,
                            };
                            machineYields.Add(machineYield);
                            index++;
                        }

                        helper.GenerateExcel(machineYields, fileName);
                        return new ObjectResult($"{FileSystemHelper.CommonFileSetting.RequestPath}/{FileSystemHelper.ExportFileName}/{temporaryFileName}");
                    }
                    catch (Exception e)
                    {
                        Logger.WriteLineError("Export AlarmDataForAnalysis yields failed, error: " + e.Message);
                        throw new BadRequestException(RequestEnum.ExportFailed);
                    }
                    finally
                    {
                        helper.Dispose();
                    }
                case 4:
                    helper = new EpPlusExcelHelper<AlarmDataForAnalysis4>(_machineRepository.GetProjectLanguage(ProjectId));
                    try
                    {
                        var machineYields = new List<AlarmDataForAnalysis4>();
                        var index = 1;
                        foreach (var yield in responses)
                        {
                            var machineYield = new AlarmDataForAnalysis4
                            {
                                No = index,
                                Abscissa = yield.title,
                                Total = yield.total,
                                Idle = yield.free,
                                Debug = yield.dubugging,
                                Fault = yield.fault,
                            };
                            machineYields.Add(machineYield);
                            index++;
                        }

                        helper.GenerateExcel(machineYields, fileName);
                        return new ObjectResult($"{FileSystemHelper.CommonFileSetting.RequestPath}/{FileSystemHelper.ExportFileName}/{temporaryFileName}");
                    }
                    catch (Exception e)
                    {
                        Logger.WriteLineError("Export AlarmDataForAnalysis yields failed, error: " + e.Message);
                        throw new BadRequestException(RequestEnum.ExportFailed);
                    }
                    finally
                    {
                        helper.Dispose();
                    }
                default:
                    helper = new EpPlusExcelHelper<AlarmDataForAnalysis>(_machineRepository.GetProjectLanguage(ProjectId));
                    try
                    {
                        var machineYields = new List<AlarmDataForAnalysis>();
                        var index = 1;
                        foreach (var yield in responses)
                        {
                            var machineYield = new AlarmDataForAnalysis
                            {
                                No = index,
                                Abscissa = yield.title,
                                Total = yield.total,
                                Idle = yield.free,
                                Debug = yield.dubugging,
                                Fault = yield.fault
                            };
                            machineYields.Add(machineYield);
                            index++;
                        }

                        helper.GenerateExcel(machineYields, fileName);
                        return new ObjectResult($"{FileSystemHelper.CommonFileSetting.RequestPath}/{FileSystemHelper.ExportFileName}/{temporaryFileName}");
                    }
                    catch (Exception e)
                    {
                        Logger.WriteLineError("Export AlarmDataForAnalysis yields failed, error: " + e.Message);
                        throw new BadRequestException(RequestEnum.ExportFailed);
                    }
                    finally
                    {
                        helper.Dispose();
                    }
            }
            throw new BadRequestException(RequestEnum.ExportFailed);
        }

        private ResponseMachineErrorSummary AddUpTimeByType(int errorType, string typeName,
            IEnumerable<CncSliceSate> sates, DateTime startTime,
            DateTime endTime, IEnumerable<ResponseGetMachineSetting> statusSettings)
        {
            double free = 0;
            double debugging = 0;
            double fault = 0;
            double freeSet = 0;
            double debugSet = 0;
            double faultSet = 0;
            var response = new ResponseMachineErrorSummary { title = typeName, dubugging = 0, free = 0, fault = 0 };
            foreach (var sate in sates)
            {
                var start = sate.StartTime;
                var end = sate.EndTime;
                if (start < startTime)
                {
                    start = startTime;
                }

                if (end > endTime)
                {
                    end = endTime;
                }

                var statusSetting = statusSettings.FirstOrDefault(q => q.machineId == sate.MachineID);
                if (statusSetting != null)
                {
                    freeSet = statusSetting.free * 60;
                    debugSet = statusSetting.debugging * 60;
                    faultSet = statusSetting.fault * 60;
                }

                switch (errorType)
                {
                    case 0:
                        if (sate.Status == 2 && debugSet > 0)
                        {
                            debugging += (end - start).TotalSeconds > debugSet ? (end - start).TotalSeconds : 0;
                        }

                        if (sate.Status == 3 && freeSet > 0)
                        {
                            free += (end - start).TotalSeconds > freeSet ? (end - start).TotalSeconds : 0;
                        }

                        if (sate.Status == 4 && faultSet > 0)
                        {
                            fault += (end - start).TotalSeconds > faultSet ? (end - start).TotalSeconds : 0;
                        }

                        break;
                    case 2: //调试
                        if (sate.Status == 2 && debugSet > 0)
                        {
                            debugging += (end - start).TotalSeconds > debugSet ? (end - start).TotalSeconds : 0;
                        }

                        break;
                    case 3: //空闲
                        if (sate.Status == 3 && freeSet > 0)
                        {
                            free += (end - start).TotalSeconds > freeSet ? (end - start).TotalSeconds : 0;
                        }

                        break;
                    case 4: //报警
                        if (sate.Status == 4 && faultSet > 0)
                        {
                            fault += (end - start).TotalSeconds > faultSet ? (end - start).TotalSeconds : 0;
                        }

                        break;
                }
            }

            response.dubugging = Math.Round(debugging / 3600, 2);
            response.fault = Math.Round(fault / 3600, 2);
            response.free = Math.Round(free / 3600, 2);
            response.total = Math.Round(response.dubugging + response.fault + response.free, 2);

            return response;
        }

        private IEnumerable<ResponseMachineErrorSummary> AddUpTimeByTime(DateTime startTime, DateTime endTime,
            int num, int errorType, IEnumerable<CncSliceSate> exceptions,
            IEnumerable<ResponseGetMachineSetting> statusSettings)
        {
            var list = new List<ResponseMachineErrorSummary>();
            if (num == 30) //按月统计
            {
                var dateTimes = DateTimeHelper.GetStrDateTimes(startTime, endTime, 2);
                foreach (var dateTime in dateTimes)
                {
                    var dtCompareStart = DateTime.Parse(dateTime + "-01");
                    var dtCompareEnd = dtCompareStart.AddMonths(1).AddSeconds(-1);

                    var query = exceptions.Where(q => q.StartTime >= dtCompareStart && q.EndTime <= dtCompareEnd);

                    var response = AddUpTimeByType(errorType, dateTime, query, dtCompareStart, dtCompareEnd,
                        statusSettings);
                    list.Add(response);
                }
            }
            else if (num == 7)
            {
                var xData2 = new List<string>();
                var xData = DateTimeHelper.GetStrDateTimes(startTime, endTime, 1);
                DateTimeHelper.GetDateTimes(startTime, endTime, 1).ForEach(q =>
                {
                    xData2.Add(q.ToString(UnixTimeHelper.DateFormat));
                });
                var interal = DateTimeHelper.GetInteral(1);
                for (int i = 1; i <= xData2.Count; i++)
                {
                    DateTime stimes;
                    DateTime endtime;
                    //stime = DateTime.Parse(xData2[i - 1]);
                    //end = DateTime.Parse(xData2[i - 1]).AddDays(interal).AddSeconds(-1);
                    var weekStart = DateTimeHelper.GetWeekRangeStart(xData[i - 1]);
                    var weekEnd = DateTimeHelper.GetWeekRangeEnd(xData[i - 1]);
                    stimes = startTime > weekStart ? startTime : weekStart;
                    endtime = endTime < weekEnd ? endTime : weekEnd.AddDays(1).AddSeconds(-1);
                    var result = new List<ResponseMachineErrorSummary>();
                    for (var stime = stimes; stime.CompareTo(endtime) <= 0; stime = stime.AddDays(1))
                    {
                        var end = stime.AddDays(1).AddSeconds(-1);
                        var query = exceptions.Where(q => (q.StartTime >= stime && q.EndTime <= end)
                                                          || (q.StartTime < end && q.EndTime > end)
                                                          || (q.StartTime < stime && q.EndTime > stime)
                                                          || (q.StartTime < stime && q.EndTime > end));
                        var response = AddUpTimeByType(errorType, stime.ToString(ParameterConstant.DateFormat), query,
                            stime, end, statusSettings);
                        response.title = xData[i - 1].ToStr();
                        result.Add(response);
                    }
                    var dubugging = 0.00;
                    var fault = 0.00;
                    var free = 0.00;
                    var total = 0.00;
                    var title = "";
                    foreach (var item in result)
                    {
                        dubugging += item.dubugging;
                        fault += item.fault;
                        free += item.free;
                        total += item.total;
                        title = item.title;
                    }

                    var data = new ResponseMachineErrorSummary();
                    data.dubugging = dubugging;
                    data.fault = fault;
                    data.free = free;
                    data.total = total;
                    data.title = title;
                    list.Add(data);
                }
            }
            else
            {
                for (var stime = startTime; stime.CompareTo(endTime) <= 0; stime = stime.AddDays(num))
                {
                    var end = stime.AddDays(num).AddSeconds(-1);
                    var query = exceptions.Where(q => (q.StartTime >= stime && q.EndTime <= end)
                                                      || (q.StartTime < end && q.EndTime > end)
                                                      || (q.StartTime < stime && q.EndTime > stime)
                                                      || (q.StartTime < stime && q.EndTime > end));
                    var response = AddUpTimeByType(errorType, stime.ToString(ParameterConstant.DateFormat), query,
                        stime, end, statusSettings);
                    list.Add(response);
                }
            }

            return list;
        }

        /// <summary>
        /// 设备运行状态统计--铁姆肯
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<IActionResult> GetMachineRunningStateSummary([FromBody] RequestMachineRunningSummary request)
        {
            //xtype 1时间范围 2产线
            //x_id :xtype = 1时： 1日 2周 3月 ；xtype = 2时：产线级别
            var dateTimes = request.datetime.Split(" - ");
            if (dateTimes.Length != 2)
            {
                throw new BadRequestException(CncEnum.DateTimeFormatError);
            }

            var success = DateTime.TryParse(dateTimes[0], out var dtStart);
            if (!success)
            {
                throw new BadRequestException(CncEnum.DateTimeFormatError);
            }

            success = DateTime.TryParse(dateTimes[1], out var dtEnd);
            if (!success)
            {
                throw new BadRequestException(CncEnum.DateTimeFormatError);
            }

            dtStart = dtStart > DateTime.Now ? DateTime.Now.Date : dtStart;

            dtEnd = dtEnd.AddDays(1).AddSeconds(-1);
            dtEnd = dtEnd > DateTime.Now ? DateTime.Now : dtEnd;

            var responses = new List<MachineCanvasDataInfo>();

            var machinIds = new List<int>();
            machinIds = _machineRepository.GetNCLevelSectionMachineIds(request.sectionId, ProjectId).ToList();

            var patchId = request.x_id.ToInt();
            if (request.xtype == 0) //时间
            {
                var type = 0;
                if (patchId == 1)
                {
                    type = 11;
                }
                else if (patchId == 2)
                {
                    type = 12;
                }
                else
                {
                    type = patchId;
                }
                var interal = DateTimeHelper.GetInteral(patchId);
                var xData = DateTimeHelper.GetStrDateTimes(dtStart, dtEnd, type);
                var dates = DateTimeHelper.GetDateTimes(dtStart, dtEnd, type);
                for (int i = 0; i < dates.Count; i++)
                {
                    DateTime start;
                    DateTime end;
                    string dateTime;
                    if (patchId == 0)
                    {
                        start = dates[i];
                        end = dates[i].AddDays(interal).AddSeconds(-1) > DateTime.Now
                            ? DateTime.Now
                            : dates[i].AddDays(interal).AddSeconds(-1);
                        dateTime = end.ToString(ParameterConstant.DateFormat);
                    }
                    else if (patchId == 1)
                    {
                        dateTime = xData[i];
                        var weekStart = DateTimeHelper.GetWeekRangeStart(dateTime);
                        var weekEnd = DateTimeHelper.GetWeekRangeEnd(dateTime);
                        start = dtStart > weekStart ? dtStart : weekStart;
                        end = dtEnd < weekEnd ? dtEnd : weekEnd.AddDays(1).AddSeconds(-1);

                    }
                    else
                    {
                        start = dates[i];
                        var dts = start.AddMonths(1);
                        dts = dts.AddDays(0 - dts.Day).AddDays(1).AddSeconds(-1);
                        end = dts > dtEnd ? dtEnd.AddDays(1).AddSeconds(-1) : dts; ;//到月底
                        dateTime = end.ToString("yyyy-MM");
                    }

                    responses.Add(await SliceManager.GetMachineRunningSlice(start, end, machinIds, CompanyId, ProjectId,
                        dateTime));
                }
            }

            if (request.xtype == 1)
            {
                try
                {
                    var sections = _machineRepository.GetNCLevelSections(request.sectionId, ProjectId);
                    var levels = sections.Where(q => q.levelid == patchId).ToList();
                    if (levels.Any())
                    {
                        foreach (var levelSection in levels)
                        {
                            var machineIds = _machineRepository.GetNCLevelSectionMachineIds(levelSection.id, ProjectId);
                            var dateTime = levelSection.title;
                            responses.Add(await SliceManager.GetMachineRunningSlice(dtStart, dtEnd, machineIds,
                                CompanyId, ProjectId, dateTime));
                        }
                    }
                    else //最后一级
                    {
                        var machine = _machineRepository.GetNCMachineBySectionId(request.sectionId, ProjectId);
                        var dateTime = machine.name;
                        responses.Add(await SliceManager.GetMachineRunningSlice(dtStart, dtEnd,
                            new List<int> { machine.id }, CompanyId, ProjectId, dateTime));
                    }
                }
                catch
                {
                    throw new BadRequestException(CommonEnum.RecordNotFound);
                }
            }

            if (request.toexcel == 0)
            {
                return new ObjectResult(responses);
            }

            return ExportMachineRunningChart(responses);
        }

        [HttpPost]
        private IActionResult ExportMachineRunningChart(List<MachineCanvasDataInfo> responses)
        {
            if (!responses.Any())
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }

            var rootDir = FileSystemHelper.GetPhysicalFolders(FileSystemHelper.CommonFileSetting.PhysicalFolder, FileSystemHelper.ExportFileName);

            var temporaryFileName = $"MachineState_{DateTime.Now:yyyyMMddHHmmss}.xlsx";
            var fileName = Path.Combine(rootDir, temporaryFileName);

            var helper = new EpPlusExcelHelper<MachineRunningList>(_machineRepository.GetProjectLanguage(ProjectId));
            try
            {
                var machineYields = new List<MachineRunningList>();
                var index = 1;
                foreach (var yield in responses)
                {
                    var machineYield = new MachineRunningList
                    {
                        No = index,
                        Abscissa = yield.datetime,
                        Run = yield.data.hour1,
                        RunHour = yield.data.num1,
                        Idle = yield.data.hour2,
                        IdleHour = yield.data.num2,
                        Debug = yield.data.hour3,
                        DebugHour = yield.data.num3
                    };
                    machineYields.Add(machineYield);
                    index++;
                }

                helper.GenerateExcel(machineYields, fileName);
                return new ObjectResult($"{FileSystemHelper.CommonFileSetting.RequestPath}/{FileSystemHelper.ExportFileName}/{temporaryFileName}");
            }
            catch (Exception e)
            {
                Logger.WriteLineError("Export ExportMachineRunningChart failed, error: " + e.Message);
                throw new BadRequestException(RequestEnum.ExportFailed);
            }
            finally
            {
                helper.Dispose();
            }
        }

        [HttpGet]
        public async Task<IActionResult> MachineCanvasDataInfoTwo(string patial_code, string page_id, int projectid)
        {
            var x_id = "1";
            var sectionId = 0;
            var startTime = DateTime.Now.ToString(ParameterConstant.DateFormat);
            var endTime = DateTime.Now.ToString(ParameterConstant.DateTimeFormat);
            var pageId = page_id.ToInt();
            if (pageId == 0)
            {
                throw new ServerException(500195);
            }
            var res = _combPageConditionRepository.Get(q => q.comb_page_id == page_id && q.page_patial_code == patial_code);
            sectionId = res.condition.ToInt();
            var patch = _levelSectionRepository.Get(q => q.id == sectionId && q.projectid == projectid);
            if (patch != null)
            {
                x_id = patch.levelid.ToStr();
            }
            var sectionIdStr = "";
            //xtype - 0:时间 1：层级
            //x_ID - 0:天 1：周 2：月 xtype=1时，为levelid
            if (!DateTime.TryParse(startTime, out var dtStart))
            {
                dtStart = DateTime.Now.AddDays(-14);
            }

            if (!DateTime.TryParse(endTime, out var dtEnd))
            {
                dtEnd = DateTime.Now;
            }

            if (dtStart > dtEnd)
            {
                throw new ServerException(100158);
            }

            if (dtStart == dtEnd)
            {
                dtStart = DateTime.Parse(dtStart.ToString(ParameterConstant.DateFromZero));
                dtEnd = DateTime.Parse(dtEnd.ToString(ParameterConstant.DateToEnd));
            }

            dtEnd = dtEnd > DateTime.Now ? DateTime.Now : dtEnd;

            var machinIds = new List<int>();
            var sectionIds = string.IsNullOrWhiteSpace(sectionIdStr)
                ? new List<int>()
                : sectionIdStr.Split(',', StringSplitOptions.RemoveEmptyEntries).Select(t => t.ToInt())
                    .ToList();
            machinIds = sectionIds.Any()
                ? _machineRepository.GetNCMachineIdsBySectionIds(sectionIds, projectid).ToList()
                : _machineRepository.GetNCLevelSectionMachineIds(sectionId, projectid).ToList();

            var responses = new List<MachineCanvasDataInfo>();
            var patchId = x_id.ToInt();
            try
            {

                var sections = sectionIds.Any()
                    ? _machineRepository.GetLevelSectionsBySectionIds(sectionIds, projectid)
                    : _machineRepository.GetNCLevelSections(sectionId, projectid);
                var levels = sections.Where(q => q.levelid == patchId).ToList();
                if (levels.Any())
                {
                    foreach (var levelSection in levels)
                    {
                        var machineIds = _machineRepository.GetNCLevelSectionMachineIds(levelSection.id, projectid);
                        var dateTime = levelSection.title;
                        responses.Add(await SliceManager.GetMachineSlice(dtStart, dtEnd, machineIds, CompanyId,
                            projectid, dateTime, 2));
                    }
                }
                else //最后一级
                {
                    var machine = _machineRepository.GetNCMachineBySectionId(sectionId, projectid);
                    var dateTime = machine.name;
                    responses.Add(await SliceManager.GetMachineSlice(dtStart, dtEnd, new List<int> { machine.id },
                        CompanyId, projectid, dateTime, 2));
                }
            }
            catch
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }

            return new ObjectResult(responses);
        }

        /// <summary>
        /// 设备状态分布  数字化工厂首页  
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<IActionResult> MachineCanvasDataInfThree([FromBody] RequestGetMachineCanvasDataInfoTwo request)
        {
            //xtype - 0:时间 1：层级
            //x_ID - 0:天 1：周 2：月 xtype=1时，为levelid
            request.x_id = "0";
            request.xtype = 0;
            if (!DateTime.TryParse(request.startTime, out var dtStart))
            {
                throw new ServerException(500146);
            }
            if (!DateTime.TryParse(dtStart.AddDays(1).AddSeconds(-1).ToString(), out var dtEnd))
            {
                throw new ServerException(500146);
            }
            var sta = dtStart.ToString(ParameterConstant.DateTimeFormat);
            var endd = dtEnd.ToString(ParameterConstant.DateTimeFormat);
            if (request.frequency > 0)
            {
                var frequency = _sigerProjectShiftRepository.Get(q => q.id == request.frequency && q.status == (int)RowState.Valid && q.projectid == ProjectId);
                var starts = UnixTimeHelper.GetTimeByUnix(frequency.start_time);
                var ends = UnixTimeHelper.GetTimeByUnix(frequency.end_time);
                if (frequency.start_time < frequency.end_time)
                {
                    sta = dtStart.ToString(ParameterConstant.DateFormat) + " " + starts;
                    endd = dtEnd.ToString(ParameterConstant.DateFormat) + " " + ends;
                    DateTime.TryParse(sta, out dtStart);
                    DateTime.TryParse(endd, out dtEnd);
                }
                else
                {
                    sta = dtStart.ToString(ParameterConstant.DateFormat) + " " + starts;
                    endd = dtStart.AddDays(1).ToString(ParameterConstant.DateFormat) + " " + ends;
                    DateTime.TryParse(endd, out dtEnd);
                    DateTime.TryParse(sta, out dtStart);
                }
            }
            if (dtStart > dtEnd)
            {
                throw new ServerException(100158);
            }
            var machinIds = new List<int>();

            machinIds = _machineRepository.GetNCLevelSectionMachineIds(request.sectionId, ProjectId).ToList();

            var responses = new List<MachineCanvasDataInfo>();
            var patchId = request.x_id.ToInt();
            if (request.xtype == 0)
            {
                var type = 0;
                if (patchId == 1)
                {
                    type = 11;
                }
                else if (patchId == 2)
                {
                    type = 12;
                }
                else
                {
                    type = patchId;
                }
                var interal = DateTimeHelper.GetInteral(patchId);
                var dates = DateTimeHelper.GetDateTimes(dtStart, dtEnd, type);
                var xData = DateTimeHelper.GetStrDateTimes(dtStart, dtEnd, type);
                var count = 0;
                foreach (var date in dates)
                {
                    DateTime start;
                    DateTime end;
                    string dateTime;
                    if (patchId == 0)
                    {
                        var startTime = sta;
                        var endTime = endd;
                        startTime = dtStart.ToString(ParameterConstant.TimeFormat);
                        endTime = dtEnd.ToString(ParameterConstant.TimeFormat);

                        DateTime.TryParse(date.ToString(ParameterConstant.DateFormat) + " " + startTime, out start);

                        end = date.AddDays(interal).AddSeconds(-1) > DateTime.Now
                            ? DateTime.Now
                            : date.AddDays(interal).AddSeconds(-1);
                        if (end > dtEnd)
                        {
                            end = dtEnd;
                        }
                        dateTime = end.ToString(ParameterConstant.DateFormat);
                    }
                    else if (patchId == 1)
                    {
                        //start = date;
                        //end = date.AddDays(interal).AddSeconds(-1) > DateTime.Now
                        //    ? DateTime.Now
                        //    : date.AddDays(interal).AddSeconds(-1);
                        dateTime = xData[count];
                        var weekStart = DateTimeHelper.GetWeekRangeStart(dateTime);
                        var weekEnd = DateTimeHelper.GetWeekRangeEnd(dateTime);
                        start = dtStart > weekStart ? dtStart : weekStart;
                        end = dtEnd < weekEnd ? dtEnd.AddDays(1).AddSeconds(-1) : weekEnd.AddDays(1).AddSeconds(-1);
                    }
                    else
                    {
                        start = dates[count];
                        var dts = start.AddMonths(1);
                        dts = dts.AddDays(0 - dts.Day).AddDays(1).AddSeconds(-1);
                        end = dts > dtEnd ? dtEnd.AddDays(1).AddSeconds(-1) : dts;//到月底
                        dateTime = end.ToString("yyyy-MM");
                    }

                    responses.Add(await SliceManager.GetMachineSlice(start, end, machinIds, CompanyId, ProjectId,
                        dateTime, 2));
                    count++;
                }
            }
            return new ObjectResult(responses);
        }
    }
}