﻿using System.Collections.Generic;
using System.Linq;
using Microsoft.AspNetCore.Mvc;
using Siger.ApiCommon.Result;
using Siger.Middlelayer.Common;
using Siger.Middlelayer.Common.Extensions;
using Siger.Middlelayer.Common.Helpers;
using Siger.Middlelayer.Repository;
using Siger.Middlelayer.Repository.Data;
using Siger.Middlelayer.Repository.Entities;
using Siger.Middlelayer.Repository.Repositories.Interface;
using Siger.Middlelayer.Repository.Request;
using Siger.Middlelayer.Repository.Response;

namespace Siger.ApiConfig.Controller
{
    public class ProjectLevelController : BaseController
    {
        private readonly IUnitOfWork _unitOfWork;
        private readonly ISigerLevelRepository _levelRepository;
        private readonly ISigerProjectLevelRepository _projectLevelRepository;
        public ProjectLevelController(IUnitOfWork unitOfWork, ISigerLevelRepository levelRepository, ISigerProjectLevelRepository projectLevel)
        {
            _unitOfWork = unitOfWork;
            _projectLevelRepository = projectLevel;
            _levelRepository = levelRepository;
        }

        [HttpGet]
        public IActionResult GetProjectLevelLists(int id, int page = 1, int pagesize = PageSize)
        {
            var response = new List<ResponseGetProjectLevel>();
            var proId = id == 0 ? ProjectId : id;
            var projectLevels = _projectLevelRepository.GetPagedList(page, pagesize, q => q.status == (int)RowState.Valid && q.projectid == proId);
            foreach (var projectLevel in projectLevels.Data)
            {
                response.Add(Mapper<siger_project_level, ResponseGetProjectLevel>.Map(projectLevel));
            }

            return new PagedObjectResult(response.OrderBy(q => q.id).ToList(), projectLevels.Total, page, pagesize);
        }

        [HttpGet]
        public IActionResult GetProjectListsWithout()
        {
            var response = new List<ResponseIdTitle>();
            var projectLevels = _projectLevelRepository.GetProjectListsWithout();
            foreach (var projectLevel in projectLevels)
            {
                response.Add(Mapper<IdTitle, ResponseIdTitle>.Map(projectLevel));
            }

            return new ObjectResult(response);
        }

        [HttpGet]
        public IActionResult GetMainLevelLists()
        {
            var response = new List<ResponseIdTitle>();
            var projectLevels = _levelRepository.GetList(q => q.status == (int)RowState.Valid).OrderBy(q => q.sorting).ToList();
            foreach (var projectLevel in projectLevels)
            {
                response.Add(Mapper<siger_level, ResponseIdTitle>.Map(projectLevel));
            }

            return new ObjectResult(response);
        }

        [HttpPost]
        public IActionResult AddCustomLevel([FromBody] RequestAddProjectLevel request)
        {
            var first = request.levelarr.FirstOrDefault();
            if (first == null)
            {
                throw new BadRequestException(RequestEnum.LevelIsEmpty);
            }

            int parentid = 0;
            foreach (var level in request.levelarr)
            {
                parentid = _projectLevelRepository.InsertAndGetId(level.title, parentid, level.id, request.projectid.ToInt());
            }

            return new ObjectResult(CommonEnum.Succefull);
        }

        [HttpGet]
        public IActionResult GetCustomLevelById(int id)
        {
            var projectLevel = _projectLevelRepository.Get(id);
            if (projectLevel == null || projectLevel.status != (int)RowState.Valid)
            {
                throw new BadRequestException(RequestEnum.ProjectLevelNotFound);
            }

            var result = Mapper<siger_project_level, ResponseGetProjectLevel>.Map(projectLevel);
            return new ObjectResult(result);
        }

        [HttpPost]
        public IActionResult UpdateCustomLevel([FromBody]RequestUpdateProjectLevel request)
        {
            var projectLevel = _projectLevelRepository.Get(request.id);
            if (projectLevel == null || projectLevel.status != (int)RowState.Valid)
            {
                throw new BadRequestException(RequestEnum.ProjectLevelNotFound);
            }

            projectLevel.title = request.title;
            _projectLevelRepository.Update(projectLevel);
            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            throw new BadRequestException(CommonEnum.SystemExcetion);
        }

        [HttpPost]
        public IActionResult DeleteCustomLevel(int id)
        {
            var canDelete = _projectLevelRepository.CanDelete(id);
            if (!canDelete)
            {
                throw new BadRequestException(RequestEnum.ProjectLevelCannotDelete);
            }

            var projectLevels = _projectLevelRepository.GetList(q=>q.projectid == id && q.status == (int)RowState.Valid);
            foreach (var projectLevel in projectLevels)
            {
                projectLevel.status = (int)RowState.Invalid;
                _projectLevelRepository.Update(projectLevel);
            }

            _unitOfWork.Commit();
            return new ObjectResult(CommonEnum.Succefull);
        }
    }
}