﻿using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http.Features;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Siger.ApiCommon.Filters;
using Siger.ApiCommon.Utilities;
using Siger.Middlelayer.Common;
using Siger.Middlelayer.Log;
using Siger.Middlelayer.Redis;
using Siger.Middlelayer.Repository;
using Siger.ApiConfig.Task;
using Microsoft.Extensions.Hosting;
using Newtonsoft.Json;
using Newtonsoft.Json.Serialization;
using Newtonsoft.Json.Converters;
using Siger.Middlelayer.Common.Configuration;
using Microsoft.AspNetCore.Server.Kestrel.Core;

namespace Siger.ApiConfig
{
    public class Startup
    {
        public Startup(IConfiguration configuration)
        {
            Configuration = configuration;
        }

        public IConfiguration Configuration { get; }

        private string _corsKeyName = "Cors";

        // This method gets called by the runtime. Use this method to add services to the container.
        public void ConfigureServices(IServiceCollection services)
        {
            services.AddCors(options =>
            {
                options.AddPolicy(_corsKeyName, builder => builder.AllowAnyOrigin()
                .AllowAnyHeader().AllowAnyMethod());
                //.WithMethods("GET", "POST", "HEAD", "PUT", "DELETE", "OPTIONS")) ;
            });
            services.AddMvc(option => option.EnableEndpointRouting = false).SetCompatibilityVersion(CompatibilityVersion.Version_3_0);
            services.AddControllers().AddNewtonsoftJson(option => {
                option.SerializerSettings.ContractResolver = new CamelCasePropertyNamesContractResolver();
                option.SerializerSettings.Converters.Add(new IsoDateTimeConverter() { DateTimeFormat = "yyyy-MM-dd HH:mm:ss" });
            });

            // If using Kestrel:
            services.Configure<KestrelServerOptions>(options =>
            {
                options.AllowSynchronousIO = true;
            });

            // If using IIS:
            services.Configure<IISServerOptions>(options =>
            {
                options.AllowSynchronousIO = true;
            });

            services.AddMvc(options =>
            {
                options.Filters.Add<ModelValidateFilter>();
                options.Filters.Add<ApiExceptionFilter>();
                options.Filters.Add<ResultFilter>();
                options.Filters.Add<TokenValidateFilter>();
            });
            Logger.RegisterLogEngine(Module.Config);
            RepositoryFactory.Instance.Initialization(services);
            Swagger.AddSwagger(services, Module.Config);
            services.Configure<FormOptions>(x =>
            {
                x.ValueLengthLimit = int.MaxValue;
                x.MultipartBodyLengthLimit = int.MaxValue; // In case of multipart
            });
            var startTask = ConfigManager.GetValue("Task", "Start", 0);
            if (startTask == 1)
            {
                //Scheduler.StartUp();
            }
        }

        // This method gets called by the runtime. Use this method to configure the HTTP request pipeline.
        public void Configure(IApplicationBuilder app, IWebHostEnvironment env, IHostApplicationLifetime lifetime)
        {
            if (env.IsDevelopment())
            {
                app.UseDeveloperExceptionPage();
            }

            app.RegisterFileSystem();
            app.UseCors(_corsKeyName);
            app.RegisterSwagger(Module.Config);
            app.UseMvc();
            app.RegisterConsul(Module.Config, lifetime);
            lifetime.ApplicationStopped.Register(RedisCache.Instance.Dispose);
            app.UseHttpsRedirection();
        }
    }
}
