﻿using Microsoft.AspNetCore.Cors;
using Microsoft.AspNetCore.Mvc;
using Siger.ApiCommon.Filters;
using Siger.Middlelayer.Common;
using Siger.Middlelayer.Common.Extensions;
using Siger.Middlelayer.Redis;
using Siger.Middlelayer.Redis.RedisEntities;
using Siger.Middlelayer.Repository.Repositories.Interface;
using Siger.Middlelayer.Repository.Request;
using Siger.Middlelayer.Log;
using Siger.Middlelayer.Dapper;

namespace Siger.ApiConfig.Controller
{
    [NoTokenValidateFilter]
    [ApiController]
    [Route("config/[controller]/[action]")]
    [EnableCors("Cors")]
    public class LicenseController : ControllerBase
    {
        private readonly ISigerProjectRepository _projectRepository;
        public LicenseController(ISigerProjectRepository projectRepository)
        {
            _projectRepository = projectRepository;
        }

        [HttpGet]
        public IActionResult GetLicense(int projectId)
        {
            var project = _projectRepository.Get(q => q.id == projectId && q.status == (int) RowState.Valid);
            if (project == null)
            {
                throw new BadRequestException(RequestEnum.ProjectNotFound);
            }

            var license = RedisCache.Instance.GetLicense($"{project.companyid}_{project.id}");
            return new ObjectResult(license);
        }

        [HttpPost]
        public IActionResult SetLicense([FromBody]RequestSetLicense request)
        {
            if (string.IsNullOrWhiteSpace(request.projectId))
            {
                throw new BadRequestException(RequestEnum.IdIsEmpty);
            }
            var project = _projectRepository.Get(q => q.id == request.projectId.ToInt() && q.status == (int)RowState.Valid);
            if (project == null)
            {
                throw new BadRequestException(RequestEnum.ProjectNotFound);
            }

            var entity = new LicenseEntity
            {
                Cid = project.companyid.ToInt(),
                Pid = project.id,
                License = request.license
            };
            RedisCache.Instance.AddLicenseEntity(entity);
            return new ObjectResult(CommonEnum.Succefull);
        }

        /// <summary>
        /// 创建mysql项目表
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        public IActionResult CreateTable([FromBody]RequestCreateTabel request)
        {
            if (string.IsNullOrWhiteSpace(request.projectId))
            {
                throw new BadRequestException(RequestEnum.IdIsEmpty);
            }
            var project = _projectRepository.Get(q => q.id == request.projectId.ToInt() && q.status == (int)RowState.Valid);
            if (project == null)
            {
                throw new BadRequestException(RequestEnum.ProjectNotFound);
            }
            var pid = request.projectId.ToInt();
            var cid = project.companyid.ToInt();
            if (cid == 0 || pid == 0 || string.IsNullOrEmpty(request.sql))
            {
                throw new BadRequestException(RequestEnum.ParameterMiss);
            }
            
            RedisCache.Client.Set("DBCreateSql", request.sql);

            return new ObjectResult(CommonEnum.Succefull);
        }
    }
}