﻿using System;
using System.Collections.Generic;
using System.Linq.Expressions;
using Microsoft.AspNetCore.Mvc;
using Siger.ApiCommon.Result;
using Siger.Middlelayer.AccRepository.Entities;
using Siger.Middlelayer.AccRepository.Repositories.Interface;
using Siger.Middlelayer.AccRepository.Response;
using Siger.Middlelayer.AccRepository.Request;
using Siger.Middlelayer.Common;
using Siger.Middlelayer.Common.Extensions;
using Siger.Middlelayer.Repository.Repositories.Interface;
using Siger.Middlelayer.Repository;
using Siger.Middlelayer.Repository.Extensions;
using System.Linq;
using Siger.Middlelayer.Repository.Entities;

namespace Siger.ApiACC.Controllers
{
    public class FixtureToolsController : BaseController
    {
        private readonly IUnitOfWork _unitOfWork;
        private readonly IAutomationFixtureToolsCategoryRepository _toolsCategoryRepository;
        private readonly IAutomationFixtureToolsRepository _toolsRepository;

        public FixtureToolsController(IUnitOfWork unitOfWork, IAutomationFixtureToolsCategoryRepository toolsCategoryRepository, 
            IAutomationFixtureToolsRepository toolsRepository)
        {
            _unitOfWork = unitOfWork;
            _toolsCategoryRepository = toolsCategoryRepository;
            _toolsRepository = toolsRepository;
        }

        public IActionResult GetPageList(string category, string code, string name, string state, int page, int pagesize)
        {
            var data = _toolsRepository.GetPagedList(category.ToInt(), code, name, string.IsNullOrEmpty(state) ? -1 : state.ToInt(), ProjectId, page, pagesize);
            var list = new List<ResponseFixtureTools>();
            var categorys = _toolsCategoryRepository.GetList(q => q.projectId == ProjectId && q.status == (int)RowState.Valid).ToList();
            foreach(var item in data.Data)
            {
                var cates = GetParentCategoryList(item.cate_guid, categorys);
                var cateIds = cates.Select(q => q.id).ToList();
                cateIds.Reverse();
                item.categoryids = cateIds;
                list.Add(item);
            }
            return new PagedObjectResult(list, data.Total, page, pagesize);
        }

        private IEnumerable<siger_automation_fixture_tools_category> GetParentCategoryList(string parentId, List<siger_automation_fixture_tools_category> sections)
        {
            var query = from c in sections where c.guid == parentId select c;

            return query.ToList().Concat(query.ToList().SelectMany(t => GetParentCategoryList(t.parent, sections)));
        }

        [HttpPost]
        public IActionResult Add([FromBody]RequestAddFixtureTools req)
        {
            if(string.IsNullOrEmpty(req.partnumber) || string.IsNullOrEmpty(req.code) || string.IsNullOrEmpty(req.name) || string.IsNullOrEmpty(req.categoryid) ||
                    string.IsNullOrEmpty(req.managetype) || string.IsNullOrEmpty(req.managetype))
            {
                throw new BadRequestException(RequestEnum.ParameterMiss);
            }
            var data = _toolsRepository.Get(q => q.projectId == ProjectId && (q.name == req.name || q.code == req.code || q.partnumber == req.partnumber));
            if (data != null)
            {
                throw new BadRequestException(RequestEnum.DataExist);
            }

            var category = _toolsCategoryRepository.Get(q => q.id == req.categoryid.ToInt() && q.projectId == ProjectId && q.status == (int)RowState.Valid);
            if(category == null)
            {
                throw new BadRequestException(RequestEnum.FixtureToolCatgeoryNotFound);
            }

            var entity = new siger_automation_fixture_tools
            {
                guid = Guid.NewGuid().ToString(),
                category = category.guid,
                managetype = req.managetype.ToInt(),
                partnumber = req.partnumber,
                name = req.name,
                specification = req.specifition,
                number = req.number.ToInt(),
                remark = req.remark,
                attachment = req.fileurl,
                filename = req.filename,
                code = req.code,
                projectId = ProjectId,
                status = req.status.ToInt() == (int)RowState.Valid ? (int)RowState.Valid : (int)RowState.Invalid,
                createtime = DateTime.Now,
                updatetime = DateTime.Now,
                creator = UserId,
                updator = UserId,
            };
            _toolsRepository.Insert(entity);
            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            else
            {
                throw new BadRequestException(CommonEnum.Fail);
            }
        }

        [HttpPost]
        public IActionResult Update([FromBody]RequestUpdateFixtureTools req)
        {
            if (string.IsNullOrEmpty(req.partnumber) || string.IsNullOrEmpty(req.code) || string.IsNullOrEmpty(req.name) || string.IsNullOrEmpty(req.categoryid) ||
                    string.IsNullOrEmpty(req.managetype) || string.IsNullOrEmpty(req.managetype) || req.id <= 0)
            {
                throw new BadRequestException(RequestEnum.ParameterMiss);
            }
            var entity = _toolsRepository.Get(q => q.projectId == ProjectId && q.id == req.id);
            var data = _toolsRepository.Get(q => q.projectId == ProjectId && (q.name == req.name || q.code == req.code || q.partnumber == req.partnumber) && q.id != req.id);
            if (data != null)
            {
                throw new BadRequestException(RequestEnum.DataExist);
            }

            var category = _toolsCategoryRepository.Get(q => q.id == req.categoryid.ToInt() && q.projectId == ProjectId && q.status == (int)RowState.Valid);
            if (category == null)
            {
                throw new BadRequestException(RequestEnum.FixtureToolCatgeoryNotFound);
            }

            entity.category = category.guid;
            entity.managetype = req.managetype.ToInt();
            entity.partnumber = req.partnumber;
            entity.name = req.name;
            entity.specification = req.specifition;
            entity.number = req.number.ToInt();
            entity.remark = req.remark;
            entity.attachment = req.fileurl;
            entity.filename = req.filename;
            entity.code = req.code;
            entity.updatetime = DateTime.Now;
            entity.updator = UserId;
            entity.status = req.status.ToInt() == (int)RowState.Valid ? (int)RowState.Valid : (int)RowState.Invalid;
            _toolsRepository.Update(entity);
            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            else
            {
                throw new BadRequestException(CommonEnum.Fail);
            }
        }

        [HttpGet]
        public IActionResult Delete(int id)
        {
            var entity = _toolsRepository.Get(q => q.projectId == ProjectId && q.id == id);
            if (entity == null)
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }
            _toolsRepository.Delete(entity);
            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            else
            {
                throw new BadRequestException(CommonEnum.Fail);
            }
        }

        [HttpPost]
        public IActionResult Deletes([FromBody]RequestDeleteRange req)
        {
            if (req.ids == null || !req.ids.Any())
            {
                throw new BadRequestException(RequestEnum.ParameterMiss);
            }
            var entities = _toolsRepository.GetList(t => req.ids.Contains(t.id) && t.projectId == ProjectId).ToList();
            if (!entities.Any())
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }
            foreach (var entity in entities)
            {
                _toolsRepository.Delete(entity);
            }

            if (_unitOfWork.Commit() > 0)
                return new ObjectResult(CommonEnum.Succefull);
            throw new BadRequestException(CommonEnum.Fail);
        }

        [HttpGet]
        public IActionResult GetFxitureTooolList()
        {
            var list = _toolsRepository.GetList(q => q.projectId == ProjectId && q.status == (int)RowState.Valid).Select(
                q => new
                {
                    q.id,
                    q.code,
                    q.partnumber,
                    q.name,
                    q.specification,
                    q.number
                }).ToList();

            return new ObjectResult(list);
        }



        [HttpGet]
        public IActionResult GetCategoryPageList(string id, int page, int pagesize)
        {
            var data = _toolsCategoryRepository.GetPagedList(id.ToInt(), ProjectId, page, pagesize);
            return new PagedObjectResult(data.Data, data.Total, page, pagesize);
        }

        [HttpPost]
        public IActionResult AddCategory([FromBody]RequestAddFixtureToolsCategory req)
        {
            var data = _toolsCategoryRepository.Get(q => q.projectId == ProjectId && q.status == (int)RowState.Valid && q.name == req.name);
            if (data != null)
            {
                throw new BadRequestException(RequestEnum.DataExist);
            }
            var parent = "";
            if(req.parentid.ToInt() > 0)
            {
                var exsit = _toolsCategoryRepository.Get(q => q.projectId == ProjectId && q.status == (int)RowState.Valid && q.id == req.parentid.ToInt());
                if(exsit == null)
                {
                    throw new BadRequestException(CommonEnum.RecordNotFound);
                }
                else
                {
                    parent = exsit.guid;
                }
            }

            var entity = new siger_automation_fixture_tools_category
            {
                name = req.name,
                parent = parent,
                guid = Guid.NewGuid().ToString(),
                projectId = ProjectId,
                createtime = DateTime.Now,
                updatetime = DateTime.Now,
            };
            _toolsCategoryRepository.Insert(entity);
            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            else
            {
                throw new BadRequestException(CommonEnum.Fail);
            }
        }

        [HttpPost]
        public IActionResult UpdateCategory([FromBody]RequestUpdateFixtureToolsCategory req)
        {
            var entity = _toolsCategoryRepository.Get(q => q.projectId == ProjectId && q.status == (int)RowState.Valid && q.id == req.id);
            var data = _toolsCategoryRepository.Get(q => q.projectId == ProjectId && q.status == (int)RowState.Valid && q.name == req.name && 
                q.id != req.id);
            if (data != null)
            {
                throw new BadRequestException(RequestEnum.DataExist);
            }
            var parent = "";
            if (req.parentid.ToInt() > 0)
            {
                var exsit = _toolsCategoryRepository.Get(q => q.projectId == ProjectId && q.status == (int)RowState.Valid && q.id == req.parentid.ToInt());
                if (exsit == null)
                {
                    throw new BadRequestException(CommonEnum.RecordNotFound);
                }
                else
                {
                    parent = exsit.guid;
                }
            }

            entity.name = req.name;
            entity.parent = parent;
            entity.updatetime = DateTime.Now;
            _toolsCategoryRepository.Update(entity);
            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            else
            {
                throw new BadRequestException(CommonEnum.Fail);
            }
        }

        [HttpGet]
        public IActionResult DeleteCategory(int id)
        {
            var entity = _toolsCategoryRepository.Get(q => q.projectId == ProjectId && q.status == (int)RowState.Valid && q.id == id);
            if (entity == null)
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }

            entity.status = (int)RowState.Invalid;
            _toolsCategoryRepository.Update(entity);
            if(_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            else
            {
                throw new BadRequestException(CommonEnum.Fail);
            }
        }

        [HttpPost]
        public IActionResult DeleteCategorys([FromBody]RequestDeleteRange req)
        {
            if (req.ids == null || !req.ids.Any())
            {
                throw new BadRequestException(RequestEnum.ParameterMiss);
            }
            var entities = _toolsCategoryRepository.GetList(t =>
                req.ids.Contains(t.id) && t.projectId == ProjectId && t.status == (int)RowState.Valid).ToList();
            if (!entities.Any())
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }
            foreach (var entity in entities)
            {
                entity.status = (int)RowState.Invalid;
                _toolsCategoryRepository.Update(entity);
            }

            if (_unitOfWork.Commit() > 0)
                return new ObjectResult(CommonEnum.Succefull);
            throw new BadRequestException(CommonEnum.Fail);
        }

        [HttpGet]
        public IActionResult GetCategoryList()
        {
            var list = _toolsCategoryRepository.GetList(q => q.projectId == ProjectId && q.status == (int)RowState.Valid).ToList();
            var res = GetChildren("", list);
            return new ObjectResult(res);
        }

        private List<FixtureToolsCategoryTree> GetChildren(string parentid, List<siger_automation_fixture_tools_category> sectionDatas)
        {
            var resp = new List<FixtureToolsCategoryTree>();
            var query = sectionDatas.Where(f => f.parent == parentid);
            if (!query.Any())
            {
                return null;
            }
            foreach (var section in query)
            {
                var data = new FixtureToolsCategoryTree
                {
                    value = section.id,
                    label = section.name
                };
                data.children = GetChildren(section.guid, sectionDatas);
                resp.Add(data);
            }
            return resp;
        }
    }
}
