﻿using Microsoft.AspNetCore.Mvc;
using Siger.ApiCommon.Filters;
using Siger.ApiCommon.Result;
using Siger.Middlelayer.CncRepository.Repositories.Interface;
using Siger.Middlelayer.CncRepository.Request;
using Siger.Middlelayer.CncRepository.Response;
using Siger.Middlelayer.Common;
using Siger.Middlelayer.Common.Helpers;
using Siger.Middlelayer.Common.ModuleEnum;
using Siger.Middlelayer.Repository.Entities;
using Siger.Middlelayer.Repository.Repositories.Interface;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Siger.Middlelayer.Common.Extensions;
using Siger.Middlelayer.CncRepository.Entities;
using Siger.Middlelayer.Repository;
using Siger.Middlelayer.CncRepository.Repositories;
using Siger.Middlelayer.Dapper;
using Siger.Middlelayer.Dapper.SearchCondition;
using Siger.Middlelayer.Common.Log;
using Siger.Middlelayer.Share.Models;
using Siger.Middlelayer.Dapper.Utilities.Slice;

namespace Siger.ApiCNC.Controllers
{
    /// <summary>
    /// 工单任务一栏控制器
    /// </summary>
    public class WorkOrderListController : BaseController
    {
        private readonly IProductPlanDetailRepository _planDetailRepository;
        //private readonly ISigerProjectProductReport _sigerProjectProductReport;
        private readonly IProductRouteRepository _routeRepository;
        private readonly IProjectProductPlanDispatchReponsitory _projectProductPlanDispatchReponsitory;
        private readonly IProductPlanRepository _planRepository;
        private readonly ISigerProjectProductReport _productionReportRepository;
        private readonly ISigerProjectUserRepository _userRepository;
        private readonly IUnitOfWork _unitOfWork;
        private readonly IProductionBeatSetRepository _beatSetRepository;
        private readonly ISigerProjectMachineRepository _machineRepository;
        private readonly ISigerProjectUserRepository _projectUserRepository;
        private readonly ISigerProjectLevelSectionRepository _sigerProjectLevelSection;

        public WorkOrderListController(IProductPlanDetailRepository planDetailRepository,
            IProductRouteRepository routeRepository, IProjectProductPlanDispatchReponsitory projectProductPlanDispatchReponsitory,
            IProductPlanRepository planRepository, ISigerProjectProductReport productionReportRepository, ISigerProjectUserRepository userRepository,
            IUnitOfWork unitOfWork, IProductionBeatSetRepository beatSetRepository, ISigerProjectMachineRepository machineRepository,
            ISigerProjectUserRepository projectUserRepository,ISigerProjectLevelSectionRepository sigerProjectLevelSection)
        {
            _planDetailRepository = planDetailRepository;
          
            _routeRepository = routeRepository;
            _projectProductPlanDispatchReponsitory = projectProductPlanDispatchReponsitory;
            _planRepository = planRepository;
            _productionReportRepository = productionReportRepository;
            _userRepository = userRepository;
            _unitOfWork = unitOfWork;
            _beatSetRepository = beatSetRepository;
            _machineRepository = machineRepository;
            _projectUserRepository = projectUserRepository;
            _sigerProjectLevelSection = sigerProjectLevelSection;
        }
        /// <summary>
        /// 工令单一览
        /// </summary>
        /// <param name="productName">产品名称</param>
        /// <param name="code">工单号</param>
        /// <param name="orderNumber">订单号</param>
        /// <param name="status">工单状态</param>
        /// <param name="startTime">添加时间--开始</param>
        /// <param name="endTime">添加时间--结束</param>
        /// <param name="page"></param>
        /// <param name="pagesize"></param>
        /// <returns></returns>
        [HttpGet]
        public IActionResult GetWorkOrderList(string productName, string code, string orderNumber, int status, string startTime, string endTime, int page = 1, int pagesize = 10)
        {
            if (!string.IsNullOrWhiteSpace(startTime))
            {
                var success = DateTime.TryParse(startTime, out var dtStart);
                if (!success)
                {
                    throw new BadRequestException(CncEnum.DateTimeFormatError);
                }
            }
            if (!string.IsNullOrWhiteSpace(endTime))
            {
                var success = DateTime.TryParse(endTime, out var dtEnd);
                if (!success)
                {
                    throw new BadRequestException(CncEnum.DateTimeFormatError);
                }
            }

            var orders = _planDetailRepository.GetPagedWorkOrderListData(productName, code, orderNumber, status, startTime, endTime, ProjectId, page, pagesize);

            return new PagedObjectResult(orders.Data, orders.Total, page, pagesize);
        }

        /// <summary>
        /// 获取打印工单获取弹框数据
        /// </summary>
        /// <param name="id">工令号数据id</param>
        /// <returns></returns>
        [HttpGet]
        public IActionResult GetWorkOrderRouteDetals(int id) 
        {
            var data = _planDetailRepository.GetWorkOrderData(id,ProjectId);
            return new ObjectResult(data);
        }

        /// <summary>
        /// 查看工令单进度
        /// </summary>
        /// <param name="id">数据id</param>
        /// <returns></returns>
        [HttpGet]
        public IActionResult GetOrderProgress(int id)
        {
            var data = _planDetailRepository.GetPlanReports(ProjectId, 0, id);
            var plandts = data.GroupBy(g => g.code);
            var doneCnt = 0d;
            foreach (var dts in plandts)
            {
                doneCnt += dts.FirstOrDefault().quantity;
            }
            var result = new List<ResponseDispatchViewProgress>();
            var routeReport = data.GroupBy(g => g.route_id);
            foreach (var route in routeReport)
            {
                var obj = route.FirstOrDefault();
                var objlst = _productionReportRepository.GetList(f => f.projectid == ProjectId && f.plan_id == obj.plan_id && f.code == obj.code && f.route_id==obj.route_id);
                var stateDesc = EnumHelper.GetEnumDesc((PlanDispatch)obj.status);
                double okttl = objlst.Sum(f => f.actual_output);
                double ngttl = objlst.Sum(f => f.nok_number);
                double processPercent = (okttl + ngttl) *100 / doneCnt;
                var productionschedule = Math.Round(processPercent,2);
                result.Add(new ResponseDispatchViewProgress
                {
                    producted_number = obj.order_installCnt,
                    description = obj.route_name,
                    name = obj.route_name,
                    job_count = doneCnt,
                    ok_number = okttl,
                    nok_number = ngttl,
                    serialNumber = route.Key,
                    state = stateDesc,
                    productionschedule = productionschedule

                });

            }
            return new ObjectResult(result);
        }
        /// <summary>
        /// 根据工令单+路由 获取报工信息数据汇总app
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        public IActionResult GetWorkOrdersDetailByRoute([FromBody]RequestAppWorkOrderListByRoute request)
        {
            var planDetail = _planDetailRepository.Get(m => m.orderNumber == request.orderno && m.status != (int)RowState.Invalid && m.projectId == ProjectId);
            if (planDetail == null)
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }
            var plan = _planRepository.Get(m => m.id == planDetail.planId);
            if (plan == null)
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }
            //查对应得工序--2.0是根据工序报工
            var route = _routeRepository.Get(q => q.id == request.route_id.ToInt() && q.projectId == ProjectId && q.status == (int)RowState.Valid);
            if (route == null)
                throw new BadRequestException(CommonEnum.RecordNotFound);

            var reports = _productionReportRepository.GetList(q => q.route_id == request.route_id.ToInt() &&
             q.code == request.orderno  && q.projectid == ProjectId && q.status == (int)RowState.Valid);
            if (reports == null)
                throw new BadRequestException(CommonEnum.RecordNotFound);
            var response = new ResponseAppGetWorkOrdersDetail
            {
                product_code = plan.product_code,
                product_name = plan.product_name,
                product_drawcode=plan.draw_number,
                quantity = planDetail.quantity,
                producted_number = reports.Sum(m => m.nok_number + m.actual_output),
                ok_number = reports.Sum(m => m.actual_output),
                nok_number = reports.Sum(m => m.nok_number),
                plan_endtime = UnixTimeHelper.ConvertIntDate(plan.plan_endtime),
                ordernumber = plan.ordernumber,
                route_id = route != null ? route.id : 0,
                route_name = route != null ? route.name : "",
            };
            return new ObjectResult(response);
        }
        /// <summary>
        /// 根据工令单+路由 + 产线层级 获取报工信息数据汇总Web 
        /// </summary>
        /// <param name="request">产线层级对象</param>
        /// <returns></returns>
        [HttpPost]
        public IActionResult GetWorkOrdersDetail([FromBody]RequestAppWorkOrderListByRoute request)
        {
            var planDetail = _planDetailRepository.Get(m => m.orderNumber == request.orderno && m.status != (int)RowState.Invalid && m.projectId == ProjectId);
            if (planDetail == null)
                throw new BadRequestException(CommonEnum.RecordNotFound);
            var plan = _planRepository.Get(m => m.id == planDetail.planId);
            if (plan == null)
                throw new BadRequestException(CommonEnum.RecordNotFound);
            var levelsection = _sigerProjectLevelSection.Get(request.sectionid);
            if (levelsection==null)
                throw new BadRequestException(RequestEnum.LevelNotFound);

           
            
            //查对应得工序--2.0是根据工序报工
            var route = _routeRepository.Get(q => q.id == request.route_id.ToInt() && q.productId==plan.product_id && q.projectId == ProjectId && q.status == (int)RowState.Valid);
            if (route==null)
                throw new BadRequestException(CommonEnum.RecordNotFound);

          
            var reports = _productionReportRepository.GetList(q => q.route_id == request.route_id.ToInt() &&
             q.code == request.orderno && q.projectid == ProjectId && q.status == (int)RowState.Valid);
            if (reports == null)
                throw new BadRequestException(CommonEnum.RecordNotFound);
            var response = new ResponseAppGetWorkOrdersDetail
            {
                product_code = plan.product_code,
                sectionname = levelsection.title,
                product_name = plan.product_name,
                quantity = planDetail.quantity,
                producted_number = reports.Sum(m => m.nok_number + m.actual_output),
                ok_number = reports.Sum(m => m.actual_output),
                nok_number = reports.Sum(m => m.nok_number),
                plan_endtime = UnixTimeHelper.ConvertIntDate(plan.plan_endtime),
                ordernumber = plan.ordernumber,
                route_id = route != null ? route.id : 0,
                route_name = route != null ? route.name : "",
            };
            return new ObjectResult(response);
        }

        /// <summary>
        /// 报工
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        public async Task<IActionResult> AddProduceReport([FromBody] RequestAddProduceReport request)
        {
            if (string.IsNullOrWhiteSpace(request.code))
            {
                throw new BadRequestException(CncEnum.WorkOrderNotExist);
            }
            //工令单存在
            var planDetail = _planDetailRepository.Get(m => m.orderNumber == request.code && m.status != (int)RowState.Invalid && m.projectId == ProjectId);
            if (planDetail == null)
            {
                throw new BadRequestException(CncEnum.WorkOrderNotExist);
            }
            var machineid = _machineRepository.GetMahcineIdBySection(request.machineId.ToInt());
            if (machineid == 0)
            {
                return new ObjectResult(CncEnum.MachineIdNotExist);
            }
            //计划存在
            var plan = _planRepository.Get(q => q.id == planDetail.planId && q.projectid == ProjectId);
            if (plan == null)
            {
                throw new BadRequestException(CncEnum.WorkOrderNotExist);
            }
            if (plan.status==(int)PlanDispatch.Stop)
            {
                throw new BadRequestException(CncEnum.DataToDispatch);
            }
            //当前工令单产品对应工序
            var productRoutes = _routeRepository.GetList(q => q.productId == plan.product_id && q.projectId == ProjectId && q.status == (int)RowState.Valid);
            if (!productRoutes.Any())
            {
                throw new BadRequestException(RequestEnum.RouteNotFound);
            }
            var user = await _userRepository.GetAsync(q => q.mid == request.workercode.ToInt() && q.projectid == ProjectId);
            var entity = new siger_project_product_report
            {
                plan_id = planDetail.planId,
                product_name = plan.product_name,
                product_code = plan.product_code,
                code = request.code,
                actual_output = request.actualoutput.ToInt(),
                output= request.actualoutput.ToInt()+request.noknumber.ToInt(),
                machineid = machineid,
                station=request.machineId.ToInt(),
                nok_number = request.noknumber.ToInt(),
                worker_code = user != null ? user.work_code : "",
                worker_name = user != null ? user.name : "",
                projectid = ProjectId,
                time = UnixTimeHelper.GetNow(),
                route_name = request.route_name,
                route_id = request.route_id, //route id
                uid=request.workercode.ToInt()
            };
            if (_productionReportRepository.InsertProductReportData(entity) == 0)
            {
                throw new BadRequestException(CommonEnum.Fail);
            }
            return UpdatePlanDetailProduceNumber(plan, request.route_id, planDetail, request.actualoutput.ToInt(), request.noknumber.ToInt(), productRoutes.ToList());

           
        }
        /// <summary>
        /// 修改工令单生产数量，只有最后一道工序报工后，才累计数量
        /// </summary>
        private ActionResult UpdatePlanDetailProduceNumber(siger_project_product_plan plan, int routeid, siger_project_product_plan_detail planDetail, int actualoutput, int noknumber,List<siger_project_product_route>curRoutelist)
        {
            //查询是否是最后一道工序
            //var route = _routeRepository.GetList(q => q.productId == plan.product_id && q.projectId == ProjectId && q.status == (int)RowState.Valid).OrderByDescending(m => m.serialNumber).FirstOrDefault();
            var route = curRoutelist.OrderByDescending(m => m.serialNumber).FirstOrDefault();
            if (route == null)
            {
                throw new BadRequestException(RequestEnum.RouteNotFound);
            }
          
            planDetail.status = (int)PlanDispatch.Producing;
            plan.status = (int)PlanDispatch.Producing;
            if (route.id == routeid)
            {
                //更新工令单
                planDetail.ok_number += actualoutput;
                planDetail.nok_number += noknumber;
                planDetail.producted_number += actualoutput + noknumber;
                if (planDetail.producted_number >=planDetail.quantity)
                {
                    planDetail.status = (int)PlanDispatch.Finished;
                }
                //更新工单
                plan.producted_number += actualoutput + noknumber;
                plan.nok_number += noknumber;
                if (plan.producted_number >= plan.quantity)
                {
                    plan.status = (int)PlanDispatch.Finished;
                    plan.finish_time = UnixTimeHelper.GetNow();
                }
            }
            _planDetailRepository.Update(planDetail);
            _planRepository.Update(plan);
            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            throw new BadRequestException(CommonEnum.Fail);

        }

        private async Task<CommonEnum> UpdateProduceStatus(int planId)
        {
            var reports = await _planDetailRepository.GetListAsync(q => q.planId == planId && q.projectId == ProjectId);

            var plan = await _planRepository.GetAsync(planId);
           
            var status = PlanDispatch.HasPlan; //待生产
            var finishTime = 0;
            var totalOutput = reports.Sum(m => m.ok_number);
            var totalNoOkNumber = reports.Sum(m => m.nok_number);
            var totalProducted = reports.Sum(m => m.producted_number);
            if (totalOutput+ totalNoOkNumber >= plan.quantity)
            {
                status = PlanDispatch.Finished; //已完成
                finishTime = UnixTimeHelper.GetNow();
            }
            else
            {
                status = PlanDispatch.Producing; //生产中
            }
            plan.status = (int)status;
            plan.finish_time = finishTime;
            plan.producted_number = totalProducted;
            plan.nok_number = reports.Sum(m => m.nok_number);
            await _planRepository.UpdateAsync(plan);
          
            if (await _unitOfWork.CommitAsync() > 0)
            {
                return CommonEnum.Succefull;
            }

            return CommonEnum.Fail;
        }

        #region 手机app端接口
        /// <summary>
        /// 根据工位code查找工站
        /// </summary>
        /// <param name="request"></param>
        [HttpPost]
        public IActionResult GetWorkStation([FromBody]RequestWorkStation request)
        {
            //var machineData = _machineRepository.Get(q => q.code == request.code && q.projectid == ProjectId && q.status == (int)RowState.Valid);

            //if (machineData == null)
            //{
            throw new BadRequestException(CommonEnum.Fail);
            //}

            //var user = _projectUserRepository.Get(f=>f.id==UserId);

            //var machineList = new List<ResponseGetWorkOrderMachine>();
            //var response = new ResponseGetWorkOrderMachine
            //{
            //    id = machineData.id],
            //    title = machineData.title,
            //    code = machineData.code,
            //    username= user.name,
            //};
            //machineList.Add(response);
            //return new ObjectResult(machineList);
        }

        /// <summary>
        /// 查询项目下所有人员
        /// </summary>
        /// <param name="name">人员姓名</param>
        /// <returns></returns>
        [HttpGet]
        public IActionResult GetUserData(string name) 
        {
            var data = _projectUserRepository.GetList(q =>
                        q.projectid == ProjectId &&
                       q.status == (int)RowState.Valid)
                   .Select(m => new ResponseGetUser
                   {
                       mid = m.mid,
                       name = m.name
                   }).ToList();
            if (!string.IsNullOrWhiteSpace(name))
            {
                data = data.Where(f=>f.name.Contains(name)).ToList();
            }
            return new ObjectResult(data);
        }

        /// <summary>
        /// 根据人员ID查报工信息
        /// </summary>
        /// <param name="userId"></param>
        /// <returns></returns>
        [HttpGet]
        public IActionResult GetReportByUser(int userId) 
        {
            var data = _planRepository.GetReportDetails(userId, ProjectId);
            return new ObjectResult(data);
        }

        /// <summary>
        /// 查当前设备得状态和运行时长
        /// </summary>
        /// <param name="sectionId"></param>
        /// <param name="timeSection"></param>
        /// <returns></returns>
        [HttpGet]
        public async Task<IActionResult> GetSectionStatusTime(int sectionId, string timeSection)
        {
            var data = new ResponseSectionStatusTime();
            var sectionIds = new List<int>();
            var machineIds = _machineRepository.GetNCLevelSectionMachineIds(sectionId, ProjectId).ToList();
            machineIds = _machineRepository.GetOrderByNCMachineIds(machineIds, ProjectId).ToList();
            var machineIDs = (sectionId == 0 && !sectionIds.Any()) ? machineIds.Take(15).ToList() : machineIds;
            var beats = _beatSetRepository.GetList(q =>
                    machineIDs.Contains(q.machineID) && q.projectID == ProjectId && q.status == (int)RowState.Valid)
                .ToList();
            if (await MachineCanvasParetoInfo(machineIDs, timeSection) is Result.MachineCanvasParetoInfoResult result)
            {
                //获取程序号
                var minStartTime = DateTime.Parse(result.startTime);
                var unixStartTime = UnixTimeHelper.ConvertDataTimeLong(minStartTime);
                var repository = new ProductRepository(CompanyId, ProjectId);
                var yields = await repository.GetYieldsAsync(result.startTime, result.endTime, machineIDs);
                if (yields != null)
                {
                    var canvas = result.canvas;
                    foreach (var canva in canvas)
                    {
                        var listMsh = new List<MachineStatusHold>();
                        foreach (var hold in canva.list_msh)
                        {
                            listMsh.Add(new MachineStatusHold
                            {
                                end_time = hold.end_time,
                                machine_status = hold.machine_status,
                                start_time = hold.start_time,
                                time_span = hold.time_span,
                            });
                        }

                        var minListTime = listMsh.Min(q => q.start_time);
                        if (minListTime - unixStartTime > 5 * 60)
                        {
                            listMsh.Insert(0, new MachineStatusHold
                            {
                                end_time = minListTime,
                                machine_status = 0,
                                start_time = (int)unixStartTime,
                                time_span = minListTime - (int)unixStartTime,
                                isFake = 1
                            });
                        }
                        canva.list_msh = listMsh;
                    }
                    canvas[0].list_msh = canvas[0].list_msh.OrderByDescending(f => f.start_time).Take(1).ToList();
                    data.machine_status = canvas[0].list_msh[0].machine_status;
                    data.time_span = canvas[0].list_msh[0].time_span;
                }
            }
            else
            {
                return new ObjectResult(data);
            }
            return new ObjectResult(data);
        }

        /// <summary>
        /// 设备切片,不包含程序号
        /// </summary>
        /// <param name="machineIds"></param>
        /// <param name="timeSection"></param>
        /// <returns></returns>
        [HttpGet]
        public async Task<IActionResult> MachineCanvasParetoInfo(List<int> machineIds, string timeSection)
        {
            var result = CheckParameter(timeSection);
            var time = result.endTime.ToDateTime();
            var dtStart = result.startTime.ToDateTime();
            var dtEnd = time > DateTime.Now ? DateTime.Now : time;

            var validMachines = _machineRepository.GetNCLevelSectionNames(machineIds, ProjectId);

            var canvas =
                await SliceManager.GetMachineSliceNew(dtStart, dtEnd, machineIds, validMachines, CompanyId, ProjectId);
            result.canvas = canvas.canvas;
            result.pareto = canvas.pareto;
            result.endTime = dtEnd.ToString(ParameterConstant.DateTimeFormat);
            return result;
        }


        private Result.MachineCanvasParetoInfoResult CheckParameter(string timeSection)
        {
            if (string.IsNullOrWhiteSpace(timeSection))
            {
                throw new BadRequestException(RequestEnum.TimeSpanNotNull);
            }

            var times = timeSection.Split(" - ");
            var success = DateTime.TryParse(times[0], out var startTime);
            if (!success)
            {
                throw new ServerException(500146);
            }

            success = DateTime.TryParse(times[1], out var endTime);
            if (!success)
            {
                throw new ServerException(500146);
            }

            var result = new Result.MachineCanvasParetoInfoResult
            {
                startTime = startTime.ToString(ParameterConstant.DateTimeFormat),
                endTime = endTime.ToString(ParameterConstant.DateTimeFormat),
                total_time_span = (endTime - startTime).TotalSeconds
            };
            return result;
        }

        /// <summary>
        /// 查看具体报警信息
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        [HttpPost]
        public IActionResult GetMachineAlarmList([FromBody] RequestGetMachineAlarmList request)
        {
            try
            {
                request.page = 1;
                request.pagesize = 10;
                var cncExceptionRepository = new AlarmAnalysResultRepository(CompanyId, ProjectId);
                var pageInfo = new Pagination(request.page, request.pagesize);
                var condition = new CncAlarmAnalysResultCondition
                {
                    MachinIds = _machineRepository.GetNCLevelSectionMachineIds(request.SectionID.ToInt(), ProjectId),
                    STime = request.Start,
                    ETime = request.End,
                    AlarmCode = request.AlarmCode,
                    AlarmContent = request.AlarmContent
                };

                var result = new List<ResponseGetMachineAlarmList>();

                var exceptions = cncExceptionRepository.GetPagedAlarmAnalysResults(pageInfo, condition);
                var machineIds = exceptions.Value.Select(m => m.machineId).Distinct();

                var stations = _machineRepository.GetNCMachinSectionInfos(machineIds, ProjectId).ToList();
                foreach (var exception in exceptions.Value)
                {
                    var machineStation = stations.FirstOrDefault(q => q.machineId == exception.machineId);
                    if (machineStation == null)
                    {
                        continue;
                    }
                    var entity = new ResponseGetMachineAlarmList
                    {
                        id = exception.id,
                        MachineCode = machineStation.section_name + "-" + machineStation.station_name,
                        Code = exception.code,
                        Message = exception.message,
                        //Duration = UnixTimeHelper.GetTimeBySecond((int)(exception.Etime - exception.Stime).TotalSeconds),
                        Duration = (exception.Etime - exception.Stime).TotalSeconds.ToString(),
                        Active = exception.status,
                        StartTime = exception.Stime.ToString(ParameterConstant.DateTimeFormat),
                        EndTime = exception.Etime.ToString(ParameterConstant.DateTimeFormat),
                        FaultContent = exception.FaultContent,
                        FaultSolution = exception.FaultSolution,
                        FaultReason = exception.FaultReason,
                        IsFault = exception.IsFault
                    };

                    result.Add(entity);
                }
                if (result.Count>0)
                {
                    result = result.Where(f => f.Active == 1).ToList();
                }
                return new PagedObjectResult(result, exceptions.Key.TotalItemCount, request.page, request.pagesize);
            }
            catch (Exception e)
            {
                Logger.WriteLineError("get GetMachineAlarmList failed, error:" + e);
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }
        }

        /// <summary>
        /// 生产进度详情
        /// </summary>
        /// <param name="ordernumber">工单号</param>
        /// <returns></returns>
        [HttpGet]
        public IActionResult GetProductionScheduleDetails(string ordernumber) 
        {
            var data = _projectProductPlanDispatchReponsitory.GetProductionScheduleDetails(ordernumber,ProjectId);
            return new ObjectResult(data);
        }
        #endregion

    }
}
