﻿using System.Collections.Generic;
using Microsoft.AspNetCore.Mvc;
using Siger.ApiCommon.Exceptions;
using Siger.ApiCommon.Result;
using Siger.Middlelayer.Common;
using Siger.Middlelayer.Common.Helpers;
using Siger.Middlelayer.Repository;
using Siger.Middlelayer.Repository.Entities;
using Siger.Middlelayer.Repository.Repositories.Interface;
using Siger.Middlelayer.Repository.Request;
using Siger.Middlelayer.Repository.Response;

namespace Siger.ApiConfig.Controller
{
    
    public class ProjectTypeController : BaseController
    {
        private readonly IUnitOfWork _unitOfWork;
        private readonly ISigerProjectTypeRepository _projectTypeRepository;
        public ProjectTypeController(IUnitOfWork unitOfWork, ISigerProjectTypeRepository typeRepository)
        {
            _unitOfWork = unitOfWork;
            _projectTypeRepository = typeRepository;
        }

        [HttpGet]
        public IActionResult GetProjectTypeLists(int page = 1, int pagesize = PageSize)
        {
            var response = new List<ResponseGetProjectType>();

            var projectTypes = _projectTypeRepository.GetPagedList(page, pagesize, q => q.status == (int)RowState.Valid);
            foreach (var projectType in projectTypes.Data)
            {
                response.Add(Mapper<siger_project_type, ResponseGetProjectType>.Map(projectType));
            }

            return new PagedObjectResult(response, projectTypes.Total, page, pagesize);
        }

        [HttpGet]
        public IActionResult GetProjectTypeInfoOne(int id)
        {
            var projectType = _projectTypeRepository.Get(id);
            if (projectType == null || projectType.status != (int)RowState.Valid)
            {
                throw new BadRequestException(RequestEnum.ProjectTypeNotFound);
            }

            var result = Mapper<siger_project_type, ResponseGetProjectType>.Map(projectType);
            return new ObjectResult(result);
        }

        [HttpGet]
        public IActionResult GetProjectType()
        {
            var response = new List<ResponseGetProjectType>();

            var projectTypes = _projectTypeRepository.GetList(q => q.status == (int)RowState.Valid);
            foreach (var projectType in projectTypes)
            {
                response.Add(Mapper<siger_project_type, ResponseGetProjectType>.Map(projectType));
            }

            return new ObjectResult(response);
        }

        [HttpPost]
        public IActionResult AddProjectType([FromBody]RequestAddProjectType request)
        {
            if (!string.IsNullOrWhiteSpace(request.flag))
            {
                if (request.flag.Length > 20)
                {
                    throw new BadRequestException(RequestEnum.FlagLengthLimited);
                }
            }
            var entity = _projectTypeRepository.Get(q => q.name == request.name && q.status == (int)RowState.Valid);
            if (entity != null)
            {
                throw new BadRequestException(RequestEnum.NameHasExist);
            }
            var projectType = new siger_project_type();
            projectType.name = request.name;
            projectType.description = request.description;
            projectType.flag = request.flag;
            projectType.roleid = RoleId;

            _projectTypeRepository.Insert(projectType);

            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            throw new BadRequestException(CommonEnum.SystemExcetion);
        }

        [HttpPost]
        public IActionResult EditProjectType([FromBody]RequestUpdateProjectType request)
        {
            if (!string.IsNullOrWhiteSpace(request.flag))
            {
                if (request.flag.Length > 20)
                {
                    throw new BadRequestException(RequestEnum.FlagLengthLimited);
                }
            }
            var projectType = _projectTypeRepository.Get(request.id);
            if (projectType == null || projectType.status != (int)RowState.Valid)
            {
                throw new BadRequestException(RequestEnum.ProjectTypeNotFound);
            }
            var entity = _projectTypeRepository.Get(q => q.name == request.name && q.id != projectType.id && q.status == (int)RowState.Valid);
            if (entity != null)
            {
                throw new BadRequestException(RequestEnum.NameHasExist);
            }
            projectType.name = request.name;
            projectType.description = request.description;
            projectType.flag = request.flag;

            _projectTypeRepository.Update(projectType);

            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            throw new BadRequestException(CommonEnum.SystemExcetion);
        }

        [HttpPost]
        public IActionResult DeleteProjectType(int id)
        {
            var projectType = _projectTypeRepository.Get(id);
            if (projectType == null || projectType.status != (int)RowState.Valid)
            {
                throw new BadRequestException(RequestEnum.ProjectTypeNotFound);
            }

            projectType.status = (int)RowState.Invalid;
            _projectTypeRepository.Update(projectType);

            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            throw new BadRequestException(CommonEnum.SystemExcetion);
        }
    }
}