﻿using Microsoft.AspNetCore.Mvc;
using Siger.Middlelayer.Common;
using Siger.Middlelayer.Repository;
using System.Linq;
using Siger.Middlelayer.QmsRepository.Repositories.Interface;
using Siger.Middlelayer.QmsRepository.Request;
using Siger.Middlelayer.Common.Extensions;
using Siger.ApiCommon.Filters;
using Siger.Middlelayer.QmsRepository.Response;
using Siger.Middlelayer.QmsRepository.Entities;
using System;
using Siger.Middlelayer.Repository.Repositories.Interface;
using System.Collections.Generic;
using Siger.Middlelayer.Repository.Response;
using Siger.Middlelayer.Dapper.CheckTrace.Repostriories;
using Siger.Middlelayer.Common.ModuleEnum;
using Siger.Middlelayer.Dapper.CheckTrace.Data;

namespace Siger.ApiQMS.Controllers
{
    public class SensorDataCollectionController : BaseController
    {
        private readonly IUnitOfWork _unitOfWork;
        private readonly IInspectStandardRepository _inspectStandard;
        private readonly IPortSettingRepository _portSettingRepository;
        private readonly ISigerProjectLevelRepository _levelRepository;
        private readonly ISigerProjectLevelSectionRepository _levelSectionRepository;
        private readonly ISigerProjectProductRepository _productRepository;
        private readonly ICheckSnListRepository _snListRepository;
        private readonly ISigerTrMaterialsRepository _materialsRepository;
        public SensorDataCollectionController(IUnitOfWork unitOfWork, IInspectStandardRepository inspectStandard,
            IPortSettingRepository portSettingRepository, ISigerProjectLevelRepository levelRepository,
            ISigerProjectLevelSectionRepository levelSectionRepository, ISigerProjectProductRepository productRepository,
            ICheckSnListRepository snListRepository, ISigerTrMaterialsRepository materialsRepository)
        {
            _unitOfWork = unitOfWork;
            _inspectStandard = inspectStandard;
            _portSettingRepository = portSettingRepository;
            _levelRepository = levelRepository;
            _levelSectionRepository = levelSectionRepository;
            _productRepository = productRepository;
            _snListRepository = snListRepository;
            _materialsRepository = materialsRepository;
        }

        [HttpGet]
        [NoTokenValidateFilter]
        public IActionResult GetPageList(int page, int pagesize, int projectid)
        {
            var data = _portSettingRepository.GetDataList(projectid);
            //return new PagedObjectResult(data.Data, data.Total, page, pagesize);
            return new ObjectResult(data);
        }

        [HttpGet]
        [NoTokenValidateFilter]
        public IActionResult GetItemList(int projectid)
        {
            var items = _inspectStandard.GetList(t => t.projectid == projectid && t.status == (int)RowState.Valid)
                .Select(q => new ResponseIdNameItem { id = q.id, name = q.item, en_name = q.item_en }).ToList();
            return new ObjectResult(items);
        }

        [HttpPost]
        [NoTokenValidateFilter]
        public IActionResult Add([FromBody]RequestAddPortSetting req)
        {
            var entity = _portSettingRepository.Get(t => t.projectid == req.projectid && t.status == (int)RowState.Valid &&
                t.itemid == req.itemid);
            if(entity != null)
            {
                throw new BadRequestException(RequestEnum.InspectItemRepeat);
            }
            var model = new siger_qms_port_setting
            {
                port = req.port,
                paritycheck = req.paritycheck,
                baudrate = req.baudrate,
                databit = req.databit,
                stopbit = req.stopbit,
                itemid = req.itemid,
                projectid = req.projectid,
                status = (int)RowState.Valid,
                createtime = DateTime.Now
            };
            _portSettingRepository.Insert(model);
            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            else
            {
                throw new BadRequestException(CommonEnum.Fail);
            }
        }

        [HttpPost]
        [NoTokenValidateFilter]
        public IActionResult Update([FromBody]RequestUpdatePortSetting req)
        {
            var data = _portSettingRepository.Get(t => t.id == req.id && t.projectid == req.projectid && t.status == (int)RowState.Valid);
            if(data == null)
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }

            data.port = req.port;
            data.paritycheck = req.paritycheck;
            data.baudrate = req.baudrate;
            data.databit = req.databit;
            data.stopbit = req.stopbit;
            data.itemid = req.itemid;

            _portSettingRepository.Update(data);
            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            else
            {
                throw new BadRequestException(CommonEnum.Fail);
            }
        }

        [HttpGet]
        [NoTokenValidateFilter]
        public IActionResult Delete(int id, int projectid)
        {
            var data = _portSettingRepository.Get(t => t.id == id && t.projectid == projectid && t.status == (int)RowState.Valid);
            if (data == null)
            {
                throw new BadRequestException(CommonEnum.RecordNotFound);
            }

            data.status = (int)RowState.Invalid;
            _portSettingRepository.Update(data);
            if (_unitOfWork.Commit() > 0)
            {
                return new ObjectResult(CommonEnum.Succefull);
            }
            else
            {
                throw new BadRequestException(CommonEnum.Fail);
            }
        }

        [HttpGet]
        [NoTokenValidateFilter]
        public IActionResult GetSectionSelect(int reverselevel, int parentid, int projectid)
        {
            var levels = _levelRepository.GetList(t => t.status == (int)RowState.Valid && t.projectid == projectid).OrderByDescending(t => t.id).ToList();
            if (levels.Count < reverselevel)
            {
                throw new BadRequestException(RequestEnum.LevelNotFound);
            }
            var res = _levelSectionRepository.GetSectionIdNamesByLevel(levels[reverselevel - 1].id, parentid, projectid).ToList();
            return new ObjectResult(res);
        }

        [HttpGet]
        [NoTokenValidateFilter]
        public IActionResult GetProducts(int projectid)
        {
            var res = _productRepository.GetList(t => t.projectid == projectid && t.status == (int)RowState.Valid)
                    .Select(t =>new 
                    {
                        t.id,
                        t.name,
                        t.code
                    }).ToList();
            return new ObjectResult(res);
        }

        [HttpGet]
        [NoTokenValidateFilter]
        public IActionResult GetMaterials(int productid, int projectid)
        {
            var res = _snListRepository.GetSonMaterialsByProductId(projectid, productid).ToList();
            return new ObjectResult(res);
        }

        [HttpGet]
        [NoTokenValidateFilter]
        public IActionResult GetSectionRoutes(int productid, int sectionid, int projectid)
        {
            var res = new List<ResponseIdName>();
            var product = _productRepository.Get(t => t.id == productid && t.status == (int)RowState.Valid && t.projectid == projectid);
            if (product != null)
            {
                res = _inspectStandard.GetSectionRouteList(product.code, sectionid, projectid).ToList();
            }

            return new ObjectResult(res);
        }

        [HttpGet]
        [NoTokenValidateFilter]
        public IActionResult GetDetailList(int productid, int sectionid, int projectid)
        {
            var res = new List<ResponseSensorDataCollectionDetail>();
            var standards = _inspectStandard.GetList(t =>
                t.productid == productid && t.sectionid == sectionid && t.projectid == projectid
                && t.status == (int)RowState.Valid).ToList();
            var senSorPorts = _portSettingRepository.GetList(t => t.projectid == projectid && t.status == (int)RowState.Valid).ToList();
            foreach (var standard in standards)
            {
                var port = senSorPorts.FirstOrDefault(t => t.itemid == standard.id);
                var model = new ResponseSensorDataCollectionDetail
                {
                    id = standard.id,
                    itemid = standard.id,
                    max_value = standard.max_value,
                    min_value = standard.min_value,
                    result = "",
                    value = null,
                    item = standard.item,
                    standard = standard.standard,
                    checktype = standard.value_type,
                    portName = port?.port,
                    parity = port?.paritycheck ?? 0,
                    bautRate = port?.baudrate ?? 0,
                    dataBits = port?.databit ?? 0,
                    stopBits = port?.stopbit ?? 0
                };
                res.Add(model);
            }
            return new ObjectResult(res);
        }

        [HttpPost]
        [NoTokenValidateFilter]
        public IActionResult AddTrace([FromBody]RequestAddSensorDataCollection req)
        {
            if (req.details == null || !req.details.Any() || string.IsNullOrEmpty(req.result))
            {
                throw new BadRequestException(RequestEnum.CheckItemNotFound);
            }
            var product = _productRepository.Get(t => t.projectid == req.projectid && t.status == (int)RowState.Valid &&
                t.id == req.productid);
            if (product == null)
            {
                throw new BadRequestException(RequestEnum.ProductNotFound);
            }
            var maxLevel = _levelRepository.GetList(t => t.status == (int)RowState.Valid && t.projectid == req.projectid).Max(q => q.id);
            var section = _levelSectionRepository.Get(t => t.status == (int)RowState.Valid && t.projectid == req.projectid && t.id == req.sectionid);
            if (section == null || section.levelid != maxLevel)
            {
                throw new ServerException(1052);
            }
            var machine = _inspectStandard.GetMachineBySectionId(section.id, req.projectid);
            if (machine == null)
            {
                throw new BadRequestException(RequestEnum.SectionMachineNull);
            }

            if (string.IsNullOrEmpty(req.result))
            {
                throw new BadRequestException(RequestEnum.SelectCheckResult);
            }

            var items = _inspectStandard.GetList(t => req.details.Select(q => q.itemid).Contains(t.id)).ToList();
            foreach (var detail in req.details)
            {
                var item = items.FirstOrDefault(t => t.id == detail.itemid);
                if (item == null)
                {
                    throw new BadRequestException(RequestEnum.ParameterMiss);
                }
                if (string.IsNullOrEmpty(detail.result) && detail.checktype == (int)ValueTypeStatus.O)
                {
                    throw new BadRequestException(RequestEnum.ParameterMiss);
                }
                if (detail.value == null && detail.checktype == (int)ValueTypeStatus.V)
                {
                    throw new BadRequestException(RequestEnum.ParameterMiss);
                }
            }

            var trace = new SnTrace
            {
                MaterialID = req.materialid.ToInt(),
                ProductID = req.productid,
                SectionID = req.sectionid,
                MachineID = machine.id,
                WorkOrder = req.workorder ?? "",
                SN = req.sn ?? "",
                Result = req.result,
                UserID = UserId,
                RouteID = req.routeid.ToInt(),
                CreateTime = DateTime.Now,
                CheckType = req.checktype
            };
            var insertDetailList = new List<SnTraceDetailList>();
            foreach (var detail in req.details)
            {
                var item = items.FirstOrDefault(t => t.id == detail.itemid);
                var result = "";
                if (item.value_type == (int)ValueTypeStatus.V)
                {
                    result = (detail.value.Value <= item.max_value && detail.value.Value >= item.min_value) ? "OK" : "NG";
                }
                else
                {
                    result = detail.result;
                }
                insertDetailList.Add(new SnTraceDetailList
                {
                    TraceID = 0,
                    ItemID = detail.itemid,
                    Result = result,
                    Value = detail.value,
                    ItemName = item.item
                });
            }
            var _traceRepostory = new TraceDetailRepository(req.companyid, req.projectid);
            var res = _traceRepostory.InsertTraceDetail(trace, insertDetailList, req.projectid);
            if (res)
            {
                var snEntity = _snListRepository.Get(t => t.projectid == req.projectid && t.status == (int)RowState.Valid &&
                    t.SN == trace.SN);
                if (snEntity == null)
                {
                    var stateCode = "000";
                    var eventNoObj = _snListRepository.GetEventNoByResult(trace.SN, req.projectid);
                    if (eventNoObj != null)
                    {
                        //出站信息获取
                        var outObj = _snListRepository.GetOutStationByEventNo(eventNoObj.EventNo, trace.SectionID, req.projectid);
                        if (outObj != null)
                        {
                            stateCode = outObj.ResultStatus;
                        }
                    }
                    var material = _materialsRepository.Get(t => t.projectId == req.projectid && t.status == (int)RowState.Valid &&
                        t.id == req.materialid.ToInt());
                    var snListObj = new siger_check_sn_list
                    {
                        SN = trace.SN,
                        BatchNumber = "",
                        ProductID = trace.ProductID,
                        ProductCode = product.code,
                        LineID = section.parentid,
                        WorkOrder = trace.WorkOrder,
                        MaterialID = material?.id ?? 0,
                        PartNumber = material?.pn ?? "",
                        StateCode = stateCode,
                        CreateTime = DateTime.Now,
                        projectid = req.projectid
                    };
                    _snListRepository.Insert(snListObj);
                    _unitOfWork.Commit();
                }

                return new ObjectResult(CommonEnum.Succefull);
            }
            throw new BadRequestException(CommonEnum.Fail);
        }

        [HttpGet]
        [NoTokenValidateFilter]
        public IActionResult GetProductsBySectionId(string sectionid)
        {
            var productIds = _inspectStandard.GetList(t => t.sectionid == sectionid.ToInt() && t.projectid == ProjectId &&
                t.status == (int)RowState.Valid).Select(t => t.productid).ToList();
            var products = _productRepository.GetList(t => productIds.Contains(t.id) && t.projectid == ProjectId &&
                t.status == (int)RowState.Valid).Select(t => new
                {
                    t.id,
                    t.name,
                    t.code
                }).ToList();

            return new ObjectResult(products);
        }

        [HttpGet]
        [NoTokenValidateFilter]
        public IActionResult GetItems(int sectionid, int productid, int projectid)
        {
            var items = _inspectStandard.GetList(t => t.projectid == projectid && t.status == (int)RowState.Valid && 
                t.sectionid == sectionid && t.productid == productid)
                .Select(q => new ResponseIdNameItem { id = q.id, name = q.item, en_name = q.item_en }).ToList();
            return new ObjectResult(items);
        }

        [HttpGet]
        [NoTokenValidateFilter]
        public IActionResult GetItemSectionSelect(int reverselevel, int parentid, int projectid)
        {
            var levels = _levelRepository.GetList(t => t.status == (int)RowState.Valid && t.projectid == projectid).OrderByDescending(t => t.id).ToList();
            if (levels.Count < reverselevel)
            {
                throw new BadRequestException(RequestEnum.LevelNotFound);
            }
            var res = _levelSectionRepository.GetSectionIdNamesByLevel(levels[reverselevel - 1].id, parentid, projectid).ToList();
            if(reverselevel == 2)
            {             
                var itemSectionIds = _inspectStandard.GetList(t => t.projectid == projectid && t.status == (int)RowState.Valid)
                        .Select(t => t.sectionid).ToList();
                var channelIds = _levelSectionRepository.GetList(t => itemSectionIds.Contains(t.id)).Select(t => t.parentid).ToList();
                res = res.Where(t => channelIds.Contains(t.id)).ToList();
            }
            if(reverselevel == 1)
            {
                var itemSectionIds = _inspectStandard.GetList(t => t.projectid == projectid && t.status == (int)RowState.Valid &&
                        res.Select(q => q.id).Contains(t.sectionid)).Select(t => t.sectionid).ToList();
                res = res.Where(t => itemSectionIds.Contains(t.id)).ToList();
            }
            return new ObjectResult(res);
        }
    }
}