﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Threading.Tasks;
using Dapper;
using MySql.Data.MySqlClient;
using Siger.Middlelayer.Common.Configuration;
using Siger.Middlelayer.Redis;

namespace Siger.Middlelayer.Dapper.CheckTrace
{
    public abstract class TraceDbContext
    {
        private static string _dbConnection;
        private static string _dbMidConnection;
        private readonly int _companyId;
        private readonly int _projectId;
        protected TraceDbContext(int companyId, int projectId)
        {
            var useOneDb = ConfigManager.GetValue("DbSetting", "UseOneDb", true);
            if (useOneDb)
            {
                _dbConnection = ConfigManager.GetValue("DbSetting", "DbConnectionKM", string.Empty);
            }
            else
            {
                _dbConnection = ConfigManager.GetValue("DbClusterSetting", "ClusterConnectionKM", string.Empty);
            }
            if (string.IsNullOrEmpty(_dbConnection))
            {
                throw new Exception("DbSetting about [DbConnectionKM] not found.");
            }

            _dbMidConnection = ConfigManager.GetValue("DbSetting", "DbConnection", string.Empty);
            if (string.IsNullOrEmpty(_dbMidConnection))
            {
                throw new Exception("DbSetting about [DbConnection] not found.");
            }

            _companyId = companyId;
            _projectId = projectId;
        }

        public string GetDbName()
        {
            return $"{_companyId}_{_projectId}";
        }

        public IDbConnection GetMidDbConnection()
        {
            try
            {
                var connecctionStr = $"{_dbMidConnection}";

                var connect = new MySqlConnection(connecctionStr);
                connect.Open();
                return connect;
            }
            catch (Exception e)
            {
                throw new Exception($"connect mysql db {_dbMidConnection} failed, error:" + e);
            }
        }

        public IDbConnection GetDbConnection()
        {
            try
            {
                var dbConfig = RedisCache.Instance.GetDbName(_companyId, _projectId);

                var connecctionStr = $"{_dbConnection};database={dbConfig.MysqlDbName}";

                var connect = new MySqlConnection(connecctionStr);
                connect.Open();
                return connect;
            }
            catch (Exception e)
            {
                throw new Exception($"connect mysql db {_companyId}_{_projectId} failed, error:" + e);
            }
        }

        protected async Task<IEnumerable<T>> GetDataListAsync<T>(string sql)
        {
            return await Task.Run(() =>
            {
                using (var conn = GetDbConnection())
                {
                    var dataList = conn.QueryAsync<T>(sql);

                    return dataList;
                }
            });
        }

        protected IEnumerable<T> GetDataList<T>(string sql)
        {
            using (var conn = GetDbConnection())
            {
                var dataList = conn.Query<T>(sql);

                return dataList;
            }
        }

        protected long Count(string table)
        {
            using (var conn = GetDbConnection())
            {
                var result = conn.ExecuteScalar<long>($"select count(1) from {table}");
                return result;
            }
        }

        protected string GetMidDataBaseName()
        {
            using (var conn = GetMidDbConnection())
            {
                return conn.Database;
            }
        }

        /// <summary>
        /// 开启事务
        /// </summary>
        /// <param name="conn"></param>
        /// <returns></returns>
        public IDbTransaction BeginTransaction(IDbConnection conn)
        {
            IDbTransaction tran = conn.BeginTransaction();
            return tran;
        }

        /// <summary>
        /// 提交事务
        /// </summary>
        /// <param name="tran"></param>
        /// <param name="conn"></param>
        public void Commit(IDbTransaction tran, IDbConnection conn)
        {
            tran.Commit();
        }

        /// <summary>
        /// 回滚事务
        /// </summary>
        /// <param name="tran"></param>
        /// <param name="conn"></param>
        public void Rollback(IDbTransaction tran, IDbConnection conn)
        {
            tran.Rollback();
        }

        public abstract void Dispose();
    }
}
